/*
             Common Public License Version 0.5

             THE ACCOMPANYING PROGRAM IS PROVIDED UNDER THE TERMS OF
             THIS COMMON PUBLIC LICENSE ("AGREEMENT"). ANY USE,
             REPRODUCTION OR DISTRIBUTION OF THE PROGRAM CONSTITUTES
             RECIPIENT'S ACCEPTANCE OF THIS AGREEMENT.

             1. DEFINITIONS

             "Contribution" means: 
                   a) in the case of the initial Contributor, the
                   initial code and documentation distributed under
                   this Agreement, and 

                   b) in the case of each subsequent Contributor:
                   i) changes to the Program, and
                   ii) additions to the Program;

                   where such changes and/or additions to the Program
                   originate from and are distributed by that
                   particular Contributor. A Contribution 'originates'
                   from a Contributor if it was added to the Program
                   by such Contributor itself or anyone acting on such
                   Contributor's behalf. Contributions do not include
                   additions to the Program which: (i) are separate
                   modules of software distributed in conjunction with
                   the Program under their own license agreement, and
                   (ii) are not derivative works of the Program.

             "Contributor" means any person or entity that distributes
             the Program.

             "Licensed Patents " mean patent claims licensable by a
             Contributor which are necessarily infringed by the use or
             sale of its Contribution alone or when combined with the
             Program. 

             "Program" means the Contributions distributed in
             accordance with this Agreement.

             "Recipient" means anyone who receives the Program under
             this Agreement, including all Contributors.

             2. GRANT OF RIGHTS

                   a) Subject to the terms of this Agreement, each
                   Contributor hereby grants Recipient a
                   non-exclusive, worldwide, royalty-free copyright
                   license to reproduce, prepare derivative works of,
                   publicly display, publicly perform, distribute and
                   sublicense the Contribution of such Contributor, if
                   any, and such derivative works, in source code and
                   object code form.

                   b) Subject to the terms of this Agreement, each
                   Contributor hereby grants Recipient a
                   non-exclusive, worldwide, royalty-free patent
                   license under Licensed Patents to make, use, sell,
                   offer to sell, import and otherwise transfer the
                   Contribution of such Contributor, if any, in source
                   code and object code form. This patent license
                   shall apply to the combination of the Contribution
                   and the Program if, at the time the Contribution is
                   added by the Contributor, such addition of the
                   Contribution causes such combination to be covered
                   by the Licensed Patents. The patent license shall
                   not apply to any other combinations which include
                   the Contribution. No hardware per se is licensed
                   hereunder.

                   c) Recipient understands that although each
                   Contributor grants the licenses to its
                   Contributions set forth herein, no assurances are
                   provided by any Contributor that the Program does
                   not infringe the patent or other intellectual
                   property rights of any other entity. Each
                   Contributor disclaims any liability to Recipient
                   for claims brought by any other entity based on
                   infringement of intellectual property rights or
                   otherwise. As a condition to exercising the rights
                   and licenses granted hereunder, each Recipient
                   hereby assumes sole responsibility to secure any
                   other intellectual property rights needed, if any.

                   For example, if a third party patent license is
                   required to allow Recipient to distribute the
                   Program, it is Recipient's responsibility to
                   acquire that license before distributing the
                   Program.

                   d) Each Contributor represents that to its
                   knowledge it has sufficient copyright rights in its
                   Contribution, if any, to grant the copyright
                   license set forth in this Agreement.

             3. REQUIREMENTS

             A Contributor may choose to distribute the Program in
             object code form under its own license agreement, provided
             that:
                   a) it complies with the terms and conditions of
                   this Agreement; and

                   b) its license agreement:
                   i) effectively disclaims on behalf of all
                   Contributors all warranties and conditions, express
                   and implied, including warranties or conditions of
                   title and non-infringement, and implied warranties
                   or conditions of merchantability and fitness for a
                   particular purpose;

                   ii) effectively excludes on behalf of all
                   Contributors all liability for damages, including
                   direct, indirect, special, incidental and
                   consequential damages, such as lost profits;

                   iii) states that any provisions which differ from
                   this Agreement are offered by that Contributor
                   alone and not by any other party; and

                   iv) states that source code for the Program is
                   available from such Contributor, and informs
                   licensees how to obtain it in a reasonable manner
                   on or through a medium customarily used for
                   software exchange.

             When the Program is made available in source code form:
                   a) it must be made available under this Agreement;
                   and
                   b) a copy of this Agreement must be included with
                   each copy of the Program. 

             Contributors may not remove or alter any copyright notices
             contained within the Program.

             Each Contributor must identify itself as the originator of
             its Contribution, if any, in a manner that reasonably
             allows subsequent Recipients to identify the originator of
             the Contribution. 

             4. COMMERCIAL DISTRIBUTION

             Commercial distributors of software may accept certain
             responsibilities with respect to end users, business
             partners and the like. While this license is intended to
             facilitate the commercial use of the Program, the
             Contributor who includes the Program in a commercial
             product offering should do so in a manner which does not
             create potential liability for other Contributors.
             Therefore, if a Contributor includes the Program in a
             commercial product offering, such Contributor ("Commercial
             Contributor") hereby agrees to defend and indemnify every
             other Contributor ("Indemnified Contributor") against any
             losses, damages and costs (collectively "Losses") arising
             from claims, lawsuits and other legal actions brought by a
             third party against the Indemnified Contributor to the
             extent caused by the acts or omissions of such Commercial
             Contributor in connection with its distribution of the
             Program in a commercial product offering. The obligations
             in this section do not apply to any claims or Losses
             relating to any actual or alleged intellectual property
             infringement. In order to qualify, an Indemnified
             Contributor must: a) promptly notify the Commercial
             Contributor in writing of such claim, and b) allow the
             Commercial Contributor to control, and cooperate with the
             Commercial Contributor in, the defense and any related
             settlement negotiations. The Indemnified Contributor may
             participate in any such claim at its own expense.

             For example, a Contributor might include the Program in a
             commercial product offering, Product X. That Contributor
             is then a Commercial Contributor. If that Commercial
             Contributor then makes performance claims, or offers
             warranties related to Product X, those performance claims
             and warranties are such Commercial Contributor's
             responsibility alone. Under this section, the Commercial
             Contributor would have to defend claims against the other
             Contributors related to those performance claims and
             warranties, and if a court requires any other Contributor
             to pay any damages as a result, the Commercial Contributor
             must pay those damages.

             5. NO WARRANTY

             EXCEPT AS EXPRESSLY SET FORTH IN THIS AGREEMENT, THE
             PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT
             WARRANTIES OR CONDITIONS OF ANY KIND, EITHER EXPRESS OR
             IMPLIED INCLUDING, WITHOUT LIMITATION, ANY WARRANTIES OR
             CONDITIONS OF TITLE, NON-INFRINGEMENT, MERCHANTABILITY OR
             FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is solely
             responsible for determining the appropriateness of using
             and distributing the Program and assumes all risks
             associated with its exercise of rights under this
             Agreement, including but not limited to the risks and
             costs of program errors, compliance with applicable laws,
             damage to or loss of data, programs or equipment, and
             unavailability or interruption of operations. 

             6. DISCLAIMER OF LIABILITY
             EXCEPT AS EXPRESSLY SET FORTH IN THIS AGREEMENT, NEITHER
             RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY
             FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
             OR CONSEQUENTIAL DAMAGES (INCLUDING WITHOUT LIMITATION
             LOST PROFITS), HOWEVER CAUSED AND ON ANY THEORY OF
             LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
             (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
             OF THE USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE
             OF ANY RIGHTS GRANTED HEREUNDER, EVEN IF ADVISED OF THE
             POSSIBILITY OF SUCH DAMAGES.

             7. GENERAL

             If any provision of this Agreement is invalid or
             unenforceable under applicable law, it shall not affect
             the validity or enforceability of the remainder of the
             terms of this Agreement, and without further action by the
             parties hereto, such provision shall be reformed to the
             minimum extent necessary to make such provision valid and
             enforceable.

             If Recipient institutes patent litigation against a
             Contributor with respect to a patent applicable to
             software (including a cross-claim or counterclaim in a
             lawsuit), then any patent licenses granted by that
             Contributor to such Recipient under this Agreement shall
             terminate as of the date such litigation is filed. In
             addition, If Recipient institutes patent litigation
             against any entity (including a cross-claim or
             counterclaim in a lawsuit) alleging that the Program
             itself (excluding combinations of the Program with other
             software or hardware) infringes such Recipient's
             patent(s), then such Recipient's rights granted under
             Section 2(b) shall terminate as of the date such
             litigation is filed.

             All Recipient's rights under this Agreement shall
             terminate if it fails to comply with any of the material
             terms or conditions of this Agreement and does not cure
             such failure in a reasonable period of time after becoming
             aware of such noncompliance. If all Recipient's rights
             under this Agreement terminate, Recipient agrees to cease
             use and distribution of the Program as soon as reasonably
             practicable. However, Recipient's obligations under this
             Agreement and any licenses granted by Recipient relating
             to the Program shall continue and survive. 

             Everyone is permitted to copy and distribute copies of
             this Agreement, but in order to avoid inconsistency the
             Agreement is copyrighted and may only be modified in the
             following manner. The Agreement Steward reserves the right
             to publish new versions (including revisions) of this
             Agreement from time to time. No one other than the
             Agreement Steward has the right to modify this Agreement.

             IBM is the initial Agreement Steward. IBM may assign the
             responsibility to serve as the Agreement Steward to a
             suitable separate entity. Each new version of the
             Agreement will be given a distinguishing version number.
             The Program (including Contributions) may always be
             distributed subject to the version of the Agreement under
             which it was received. In addition, after a new version of
             the Agreement is published, Contributor may elect to
             distribute the Program (including its Contributions) under
             the new version. Except as expressly stated in Sections
             2(a) and 2(b) above, Recipient receives no rights or
             licenses to the intellectual property of any Contributor
             under this Agreement, whether expressly, by implication,
             estoppel or otherwise. All rights in the Program not
             expressly granted under this Agreement are reserved.

             This Agreement is governed by the laws of the State of New
             York and the intellectual property laws of the United
             States of America. No party to this Agreement will bring a
             legal action under this Agreement more than one year after
             the cause of action arose. Each party waives its rights to
             a jury trial in any resulting litigation. 


*/

/* (C) COPYRIGHT International Business Machines Corp. 2001,2002          */

// loadsave.c
//
// routines associated with loading/saving files
//
//
#define _GNU_SOURCE
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>
#include <alloca.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ipc.h>
#include <sys/file.h>
#include <errno.h>
#include <syslog.h>
#include <pwd.h>
#include <grp.h>
#include "pkcs11types.h"
#include "defs.h"
#include "host_defs.h"
#include "h_extern.h"
#include "tok_spec_struct.h"
#include "pkcs32.h"
#include "sw_crypt.h"
#include "trace.h"

/* #include "../api/apiproto.h" */

char *pk_dir;

CK_BYTE *get_pk_dir(char *fname)
{
	struct passwd *pw = NULL;

	if (token_specific.data_store.per_user &&
	    (pw = getpwuid(getuid())) != NULL)
		sprintf(fname,"%s/%s", pk_dir, pw->pw_name);
	else
		sprintf(fname, "%s", pk_dir);

	return fname;
}

static CK_RV get_encryption_info_for_clear_key(CK_ULONG *p_key_len,
					       CK_ULONG *p_block_size)
{
	CK_ULONG key_len = 0L;
	CK_ULONG block_size = 0L;

	switch (token_specific.data_store.encryption_algorithm) {
	case CKM_DES3_CBC:
		key_len = 3 * DES_KEY_SIZE;
		block_size = DES_BLOCK_SIZE;
		break;
	case CKM_AES_CBC:
		key_len = AES_KEY_SIZE_256;
		block_size = AES_BLOCK_SIZE;
		break;
	default:
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
		return ERR_MECHANISM_INVALID;
	}

	if (p_key_len)
		*p_key_len = key_len;
	if (p_block_size)
		*p_block_size = block_size;

	return CKR_OK;
}

static CK_RV get_encryption_info(CK_ULONG *p_key_len,
				 CK_ULONG *p_block_size)
{
	CK_RV rc;

	rc = get_encryption_info_for_clear_key(p_key_len, p_block_size);
	if (rc != CKR_OK)
		return rc;

	/* Tokens that use a secure key have a different size for key because
	 * it's just an indentifier not a real key. token_keysize > 0 indicates
	 * that a token uses a specific key format.
	 */
	if (token_specific.token_keysize) {
		if (p_key_len)
			*p_key_len = token_specific.token_keysize;
	}
	return CKR_OK;
}

static CK_BYTE *duplicate_initial_vector(const CK_BYTE *iv)
{
	CK_ULONG block_size = 0L;
	CK_BYTE *initial_vector = NULL;

	if (iv == NULL)
		goto done;

	if (get_encryption_info(NULL, &block_size) != CKR_OK)
		goto done;

	initial_vector = malloc(block_size);
	if (initial_vector == NULL) {
		goto done;
	}
	memcpy(initial_vector, iv, block_size);

done:
	return initial_vector;
}

static CK_RV encrypt_data(CK_BYTE *key, CK_ULONG keylen, const CK_BYTE *iv,
			  CK_BYTE *clear, CK_ULONG clear_len,
			  CK_BYTE *cipher, CK_ULONG *p_cipher_len)
{
#ifndef  CLEARTEXT
	CK_RV rc = CKR_OK;
	CK_BYTE *initial_vector = NULL;
	OBJECT *keyobj = NULL;
	CK_KEY_TYPE     keyType;
	CK_OBJECT_CLASS keyClass = CKO_SECRET_KEY;
	CK_ATTRIBUTE key_tmpl[] =
	{
		{CKA_CLASS, &keyClass, sizeof(keyClass)},
		{CKA_KEY_TYPE, &keyType, sizeof(keyType)},
		{CKA_VALUE, key, keylen}
	};

	switch (token_specific.data_store.encryption_algorithm) {
	case CKM_DES3_CBC:
		keyType = CKK_DES3;
		break;
	case CKM_AES_CBC:
		keyType = CKK_AES;
		break;
	default:
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
		return ERR_MECHANISM_INVALID;
	}
	rc = object_create_skel(key_tmpl, 3, MODE_CREATE, CKO_SECRET_KEY,
				keyType, &keyobj);
	if (rc != CKR_OK) {
		TRACE_DEVEL("object_create_skel failed.\n");
		return rc;
	}

	initial_vector = duplicate_initial_vector(iv);
	if (initial_vector == NULL) {
		TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
		return ERR_HOST_MEMORY;
	}

	switch (token_specific.data_store.encryption_algorithm) {
	case CKM_DES3_CBC:
		rc = ckm_des3_cbc_encrypt(clear, clear_len,
					  cipher, p_cipher_len,
					  initial_vector, keyobj);
		break;
	case CKM_AES_CBC:
		rc = ckm_aes_cbc_encrypt(clear, clear_len,
		                         cipher, p_cipher_len,
					 initial_vector, keyobj);
		break;
	default:
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
		rc = ERR_MECHANISM_INVALID;
	}

	if (initial_vector)
		free(initial_vector);

	return rc;

#else
	memcpy(cipher, clear, clear_len);
	return CKR_OK;
#endif
}

static CK_RV encrypt_data_with_clear_key(CK_BYTE *key, CK_ULONG keylen,
					 const CK_BYTE *iv,
					 CK_BYTE *clear, CK_ULONG clear_len,
					 CK_BYTE *cipher, CK_ULONG *p_cipher_len)
{
#ifndef CLEARTEXT
	CK_RV rc = CKR_OK;
	CK_BYTE *initial_vector = NULL;

	/* If token doesn't have a specific key size that means that it uses a
	 * clear key.
	 */
	if (token_specific.token_keysize == 0) {
		return encrypt_data(key, keylen, iv, clear, clear_len,
				    cipher, p_cipher_len);
	}

	/* Fall back to a software alternative if key is secure. */
	initial_vector = duplicate_initial_vector(iv);
	if (initial_vector == NULL) {
		TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
		return ERR_HOST_MEMORY;
	}

	switch (token_specific.data_store.encryption_algorithm) {
	case CKM_DES3_CBC:
		rc = sw_des3_cbc_encrypt(clear, clear_len,
					 cipher, p_cipher_len,
					 initial_vector, key);
		break;
	default:
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
		rc = ERR_MECHANISM_INVALID;
	}

	if (initial_vector)
		free(initial_vector);

	return rc;

#else
	memcpy(cipher, clear, clear_len);
	return CKR_OK;
#endif
}

static CK_RV decrypt_data(CK_BYTE *key, CK_ULONG keylen, const CK_BYTE *iv,
			  CK_BYTE *cipher, CK_ULONG cipher_len,
			  CK_BYTE *clear, CK_ULONG *p_clear_len)
{
#ifndef  CLEARTEXT
	CK_RV rc = CKR_OK;
	CK_BYTE *initial_vector = NULL;
	OBJECT *keyobj = NULL;
	CK_KEY_TYPE     keyType;
	CK_OBJECT_CLASS keyClass = CKO_SECRET_KEY;
	CK_ATTRIBUTE key_tmpl[] =
	{
		{CKA_CLASS, &keyClass, sizeof(keyClass)},
		{CKA_KEY_TYPE, &keyType, sizeof(keyType)},
		{CKA_VALUE, key, keylen}
	};

	switch (token_specific.data_store.encryption_algorithm) {
	case CKM_DES3_CBC:
		keyType = CKK_DES3;
		break;
	case CKM_AES_CBC:
		keyType = CKK_AES;
		break;
	default:
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
		return ERR_MECHANISM_INVALID;
	}
	rc = object_create_skel(key_tmpl, 3, MODE_CREATE, CKO_SECRET_KEY,
				keyType, &keyobj);
	if (rc != CKR_OK) {
		TRACE_DEVEL("object_create_skel failed.\n");
		return rc;
	}

	initial_vector = duplicate_initial_vector(iv);
	if (initial_vector == NULL) {
		TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
		return ERR_HOST_MEMORY;
	}

	switch (token_specific.data_store.encryption_algorithm) {
	case CKM_DES3_CBC:
		rc = ckm_des3_cbc_decrypt(cipher, cipher_len,
					  clear, p_clear_len,
					  initial_vector, keyobj);
		break;
	case CKM_AES_CBC:
		rc = ckm_aes_cbc_decrypt(cipher, cipher_len,
					 clear, p_clear_len,
					 initial_vector, keyobj);
		break;
	default:
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
		rc = ERR_MECHANISM_INVALID;
	}

	if (initial_vector)
		free(initial_vector);

	return rc;

#else
	memcpy(clear, cipher, cipher_len);
	return CKR_OK;
#endif
}

static CK_RV decrypt_data_with_clear_key(CK_BYTE *key, CK_ULONG keylen, 
					 const CK_BYTE *iv,
					 CK_BYTE *cipher, CK_ULONG cipher_len,
					 CK_BYTE *clear, CK_ULONG *p_clear_len)
{
#ifndef CLEARTEXT
	CK_RV rc = CKR_OK;
	CK_BYTE *initial_vector = NULL;

	/* If token doesn't have a specific key size that means that it uses a
	 * clear key.
	 */
	if (token_specific.token_keysize == 0) {
		return decrypt_data(key, keylen, iv, cipher, cipher_len,
				    clear, p_clear_len);
	}

	/* Fall back to a software alternative if key is secure. */
	initial_vector = duplicate_initial_vector(iv);
	if (initial_vector == NULL) {
		TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
		return ERR_HOST_MEMORY;
	}

	switch (token_specific.data_store.encryption_algorithm) {
	case CKM_DES3_CBC:
		rc = sw_des3_cbc_decrypt(cipher, cipher_len, clear, p_clear_len,
					 initial_vector, key);
		break;
	default:
		TRACE_ERROR("%s\n", ock_err(ERR_MECHANISM_INVALID));
		rc = ERR_MECHANISM_INVALID;
	}

	if (initial_vector)
		free(initial_vector);

	return rc;

#else
	memcpy(clear, cipher, cipher_len);
	return CKR_OK;
#endif
}

void set_perm(int file)
{
	struct group *grp;

	if (token_specific.data_store.per_user) {
		/* In the TPM token, with per user data stores, we don't share
		 * the token object amongst a group. In fact, we want to
		 * restrict access to a single user */
		fchmod(file,S_IRUSR|S_IWUSR);
	} else {
		// Set absolute permissions or rw-rw-r--
		fchmod(file, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH);

		grp = getgrnam("pkcs11");	// Obtain the group id
		if (grp) {
			// set ownership to root, and pkcs11 group
			if (fchown(file, getuid(), grp->gr_gid) != 0) {
				goto error;
			}
		} else {
			goto error;
		}
	}

	return;

error:
	TRACE_DEVEL("Unable to set permissions on file.\n");
}

//
//
CK_RV 
load_token_data(CK_SLOT_ID slot_id)
{
	FILE *fp = NULL;
	CK_BYTE fname[PATH_MAX];
	TOKEN_DATA td;
	CK_RV rc;
	char pk_dir_buf[PATH_MAX];

	rc = XProcLock();
	if (rc != CKR_OK) {
		TRACE_ERROR("Failed to get Process Lock.\n");
		goto out_nolock;
	}

	sprintf(fname, "%s/%s", get_pk_dir(pk_dir_buf), PK_LITE_NV);
	fp = fopen((char *)fname, "r");
	if (!fp) {
		/* Better error checking added */
		if (errno == ENOENT) {
			/* init_token_data may call save_token_data, which
			 * grabs the lock, so we must release it around this
			 * call */
			XProcUnLock();
			init_token_data(slot_id);
			rc = XProcLock();
			if (rc != CKR_OK) {
				TRACE_ERROR("Failed to get Process Lock.\n");
				goto out_nolock;
			}

			fp = fopen((char *)fname, "r");
			if (!fp) {
				// were really hosed here since the created
				// did not occur
				TRACE_ERROR("fopen(%s): %s\n",
					     fname, strerror(errno));
				rc = CKR_FUNCTION_FAILED;
				goto out_unlock;
			}
		} else {
			/* Could not open file for some unknown reason */
			TRACE_ERROR("fopen(%s): %s\n", fname, strerror(errno));
			rc = CKR_FUNCTION_FAILED;
			goto out_unlock;
		}
	}
	set_perm(fileno(fp));

	/* Load generic token data */
	if (!fread(&td, sizeof(TOKEN_DATA), 1, fp)) {
		rc = CKR_FUNCTION_FAILED;
		goto out_unlock;
	}
	memcpy(nv_token_data, &td, sizeof(TOKEN_DATA));

	/* Load token-specific data */
	if (token_specific.t_load_token_data) {
		rc = token_specific.t_load_token_data(slot_id, fp);
		if (rc)
			goto out_unlock;
	}

	rc = CKR_OK;

out_unlock:
	XProcUnLock();

out_nolock:
	if (fp)
		fclose(fp);
	return rc;
}

//
//
CK_RV save_token_data(CK_SLOT_ID slot_id)
{
	FILE *fp = NULL;
	TOKEN_DATA td;
	CK_RV rc;
	CK_BYTE fname[PATH_MAX];
	char pk_dir_buf[PATH_MAX];

	rc = XProcLock();
	if (rc != CKR_OK) {
		TRACE_ERROR("Failed to get Process Lock.\n");
		goto out_nolock;
	}

	sprintf(fname, "%s/%s", get_pk_dir(pk_dir_buf), PK_LITE_NV);
	fp = fopen((char *)fname, "r+");
	if (!fp) {
		fp = fopen((char *)fname, "w");
		if (!fp) {
			TRACE_ERROR("fopen(%s): %s\n", fname, strerror(errno));
			rc = CKR_FUNCTION_FAILED;
			goto done;
		}
	}
	set_perm(fileno(fp));

	/* Write generic token data */
	memcpy(&td, nv_token_data, sizeof(TOKEN_DATA));
	if (!fwrite(&td, sizeof(TOKEN_DATA), 1, fp)) {
		rc = CKR_FUNCTION_FAILED;
		goto done;
	}

	/* Write token-specific data */
	if (token_specific.t_save_token_data) {
		rc = token_specific.t_save_token_data(slot_id, fp);
		if (rc)
			goto done;
	}

	rc = CKR_OK;

done:
	XProcUnLock();

out_nolock:
	if (fp)
		fclose(fp);
	return rc;
}

//
//
CK_RV save_token_object(OBJECT * obj)
{
	FILE *fp = NULL;
	CK_BYTE line[100];
	CK_RV rc;
	CK_BYTE fname[PATH_MAX];
	char pk_dir_buf[PATH_MAX];

	if (object_is_private(obj) == TRUE)
		rc = save_private_token_object(obj);
	else
		rc = save_public_token_object(obj);

	if (rc != CKR_OK)
		return rc;

	// update the index file if it exists
	sprintf(fname, "%s/%s/%s", get_pk_dir(pk_dir_buf), PK_LITE_OBJ_DIR,
		PK_LITE_OBJ_IDX);
	fp = fopen((char *)fname, "r");
	if (fp) {
		set_perm(fileno(fp));
		while (!feof(fp)) {
			(void)fgets((char *)line, 50, fp);
			if (!feof(fp)) {
				line[strlen(line) - 1] = 0;
				if (strcmp(line, obj->name) == 0) {
					fclose(fp);
					// object is already in the list
					return CKR_OK;
				}
			}
		}
		fclose(fp);
	}
	// we didn't find it...either the index file doesn't exist or this
	// is a new object...
	//
	fp = fopen((char *)fname, "a");
	if (!fp) {
		TRACE_ERROR("fopen(%s): %s\n", fname, strerror(errno));
		return CKR_FUNCTION_FAILED;
	}
	set_perm(fileno(fp));

	set_perm(fileno(fp));
	fprintf(fp, "%s\n", obj->name);
	fclose(fp);

	return CKR_OK;
}

// this is the same as the old version.  public token objects are stored in the
// clear
//
CK_RV save_public_token_object(OBJECT * obj)
{
	FILE *fp = NULL;
	CK_BYTE *clear = NULL;
	CK_BYTE fname[PATH_MAX];
	CK_ULONG clear_len;
	CK_BBOOL flag = FALSE;
	CK_RV rc;
	CK_ULONG_32 total_len;
	char pk_dir_buf[PATH_MAX];

	rc = object_flatten(obj, &clear, &clear_len);
	if (rc != CKR_OK) {
		goto error;
	}

	sprintf(fname, "%s/%s/", get_pk_dir(pk_dir_buf), PK_LITE_OBJ_DIR);
	strncat((char *)fname, (char *)obj->name, 8);
	fp = fopen((char *)fname, "w");
	if (!fp) {
		TRACE_ERROR("fopen(%s): %s\n", fname, strerror(errno));
		rc = CKR_FUNCTION_FAILED;
		goto error;
	}

	set_perm(fileno(fp));

	total_len = clear_len + sizeof(CK_ULONG_32) + sizeof(CK_BBOOL);

	(void)fwrite(&total_len, sizeof(CK_ULONG_32), 1, fp);
	(void)fwrite(&flag, sizeof(CK_BBOOL), 1, fp);
	(void)fwrite(clear, clear_len, 1, fp);

	fclose(fp);
	free(clear);

	return CKR_OK;

error:
	if (fp)
		fclose(fp);
	if (clear)
		free(clear);
	return rc;
}

//
//
CK_RV save_private_token_object(OBJECT * obj)
{
	FILE *fp = NULL;
	CK_BYTE *obj_data = NULL;
	CK_BYTE *clear = NULL;
	CK_BYTE *cipher = NULL;
	CK_BYTE *ptr = NULL;
	CK_BYTE fname[100];
	CK_BYTE hash_sha[SHA1_HASH_SIZE];
	CK_BYTE *key = NULL;
	CK_ULONG key_len = 0L;
	CK_ULONG block_size = 0L;
	CK_ULONG obj_data_len, clear_len, cipher_len;
	CK_ULONG padded_len;
	CK_BBOOL flag;
	CK_RV rc;
	CK_ULONG_32 obj_data_len_32;
	CK_ULONG_32 total_len;
	char pk_dir_buf[PATH_MAX];

	rc = object_flatten(obj, &obj_data, &obj_data_len);
	obj_data_len_32 = obj_data_len;
	if (rc != CKR_OK) {
		goto error;
	}
	//
	// format for the object file:
	//    private flag
	//    ---- begin encrypted part        <--+
	//       length of object data            |
	//       object data                      +---- sensitive part
	//       SHA of (object data)             |
	//    ---- end encrypted part          <--+
	//
	compute_sha1(obj_data, obj_data_len, hash_sha);

	// encrypt the sensitive object data.  need to be careful.
	// if I use the normal high-level encryption routines I'll need to
	// create a tepmorary key object containing the master key, perform the
	// encryption, then destroy the key object.  There is a race condition
	// here if the application is multithreaded (if a thread-switch occurs,
	// the other application thread could do a FindObject and be able to
	// access the master key object.
	//
	// So I have to use the low-level encryption routines.
	//

	if ((rc = get_encryption_info(&key_len, &block_size)) != CKR_OK)
		goto error;

	// Duplicate key
	key = malloc(key_len);
	if (!key)
		goto oom_error;
	memcpy(key, master_key, key_len);


	clear_len = sizeof(CK_ULONG_32) + obj_data_len_32 + SHA1_HASH_SIZE;
	cipher_len = padded_len = block_size * (clear_len / block_size + 1);

	clear = malloc(padded_len);
	cipher = malloc(padded_len);
	if (!clear || !cipher)
		goto oom_error;

	// Build data that will be encrypted
	ptr = clear;
	memcpy(ptr, &obj_data_len_32, sizeof(CK_ULONG_32));
	ptr += sizeof(CK_ULONG_32);
	memcpy(ptr, obj_data, obj_data_len_32);
	ptr += obj_data_len_32;
	memcpy(ptr, hash_sha, SHA1_HASH_SIZE);

	add_pkcs_padding(clear + clear_len, block_size, clear_len,
			 padded_len);

	rc = encrypt_data_with_clear_key(key, key_len,
			  token_specific.data_store.obj_initial_vector,
			  clear, padded_len, cipher, &cipher_len);
	if (rc != CKR_OK) {
		goto error;
	}

	sprintf(fname, "%s/%s/", get_pk_dir(pk_dir_buf), PK_LITE_OBJ_DIR);
	strncat((char *)fname, (char *)obj->name, 8);
	fp = fopen((char *)fname, "w");
	if (!fp) {
		TRACE_ERROR("fopen(%s): %s\n", fname, strerror(errno));
		rc = CKR_FUNCTION_FAILED;
		goto error;
	}

	set_perm(fileno(fp));

	total_len = sizeof(CK_ULONG_32) + sizeof(CK_BBOOL) + cipher_len;

	flag = TRUE;

	(void)fwrite(&total_len, sizeof(CK_ULONG_32), 1, fp);
	(void)fwrite(&flag, sizeof(CK_BBOOL), 1, fp);
	(void)fwrite(cipher, cipher_len, 1, fp);

	fclose(fp);
	free(obj_data);
	free(clear);
	free(cipher);
	free(key);
	return CKR_OK;

oom_error:
	rc = CKR_HOST_MEMORY;

error:
	if (fp)
		fclose(fp);
	if (obj_data)
		free(obj_data);
	if (clear)
		free(clear);
	if (cipher)
		free(cipher);
	if (key)
		free(key);

	return rc;
}

//
//
CK_RV load_public_token_objects(void)
{
	FILE *fp1 = NULL, *fp2 = NULL;
	CK_BYTE *buf = NULL;
	CK_BYTE tmp[PATH_MAX], fname[PATH_MAX], iname[PATH_MAX];
	CK_BBOOL priv;
	CK_ULONG_32 size;
	size_t read_size;
	char pk_dir_buf[PATH_MAX];

	sprintf(iname, "%s/%s/%s", get_pk_dir(pk_dir_buf), PK_LITE_OBJ_DIR,
		PK_LITE_OBJ_IDX);

	fp1 = fopen((char *)iname, "r");
	if (!fp1)
		return CKR_OK;	// no token objects

	while (!feof(fp1)) {
		(void)fgets((char *)tmp, 50, fp1);
		if (!feof(fp1)) {
			tmp[strlen((char *)tmp) - 1] = 0;

			sprintf((char *)fname, "%s/%s/", get_pk_dir(pk_dir_buf),
				PK_LITE_OBJ_DIR);
			strcat((char *)fname, (char *)tmp);

			fp2 = fopen((char *)fname, "r");
			if (!fp2)
				continue;

			fread(&size, sizeof(CK_ULONG_32), 1, fp2);
			fread(&priv, sizeof(CK_BBOOL), 1, fp2);
			if (priv == TRUE) {
				fclose(fp2);
				continue;
			}
			// size--;
			size = size - sizeof(CK_ULONG_32) - sizeof(CK_BBOOL);
			buf = (CK_BYTE *) malloc(size);
			if (!buf) {
				fclose(fp2);
				OCK_SYSLOG(LOG_ERR,
					   "Cannot malloc %u bytes to read in "
					   "token object %s (ignoring it)",
					   size, fname);
				continue;
			}

			read_size = fread(buf, 1, size, fp2);
			if (read_size != size) {
				fclose(fp2);
				free(buf);
				OCK_SYSLOG(LOG_ERR,
					   "Cannot read token object %s "
					   "(ignoring it)", fname);
				continue;
			}
			// ... grab object mutex here.
			MY_LockMutex(&obj_list_mutex);
			if (object_mgr_restore_obj_withSize(buf, NULL, size) !=
							    CKR_OK) {
				OCK_SYSLOG(LOG_ERR,
					   "Cannot restore token object %s "
					   "(ignoring it)", fname);
			}
			MY_UnlockMutex(&obj_list_mutex);
			free(buf);
			fclose(fp2);
		}
	}
	fclose(fp1);

	return CKR_OK;
}

//
//
CK_RV load_private_token_objects(void)
{
	FILE *fp1 = NULL, *fp2 = NULL;
	CK_BYTE *buf = NULL;
	CK_BYTE tmp[PATH_MAX], fname[PATH_MAX], iname[PATH_MAX];
	CK_BBOOL priv;
	CK_ULONG_32 size;
	CK_RV rc;
	size_t read_size;
	char pk_dir_buf[PATH_MAX];

	sprintf(iname, "%s/%s/%s", get_pk_dir(pk_dir_buf), PK_LITE_OBJ_DIR,
		PK_LITE_OBJ_IDX);

	fp1 = fopen((char *)iname, "r");
	if (!fp1)
		return CKR_OK;	// no token objects

	while (!feof(fp1)) {
		(void)fgets((char *)tmp, 50, fp1);
		if (!feof(fp1)) {
			tmp[strlen((char *)tmp) - 1] = 0;

			sprintf((char *)fname, "%s/%s/", get_pk_dir(pk_dir_buf),
				PK_LITE_OBJ_DIR);
			strcat((char *)fname, (char *)tmp);

			fp2 = fopen((char *)fname, "r");
			if (!fp2)
				continue;

			fread(&size, sizeof(CK_ULONG_32), 1, fp2);
			fread(&priv, sizeof(CK_BBOOL), 1, fp2);
			if (priv == FALSE) {
				fclose(fp2);
				continue;
			}
			//size--;
			size = size - sizeof(CK_ULONG_32) - sizeof(CK_BBOOL);
			buf = (CK_BYTE *) malloc(size);
			if (!buf) {
				fclose(fp2);
				OCK_SYSLOG(LOG_ERR,
					   "Cannot malloc %u bytes to read in "
					   "token object %s (ignoring it)",
					   size, fname);
				continue;
			}

			read_size = fread((char *)buf, 1, size, fp2);
			if (read_size != size) {
				free(buf);
				fclose(fp2);
				OCK_SYSLOG(LOG_ERR,
					   "Cannot read token object %s "
					   "(ignoring it)", fname);
				continue;
			}
			// Grab object list  mutex
			MY_LockMutex(&obj_list_mutex);
			rc = restore_private_token_object(buf, size, NULL);
			MY_UnlockMutex(&obj_list_mutex);
			if (rc != CKR_OK)
				goto error;

			free(buf);
			fclose(fp2);
		}
	}
	fclose(fp1);

	return CKR_OK;

error:
	if (buf)
		free(buf);
	if (fp1)
		fclose(fp1);
	if (fp2)
		fclose(fp2);
	return rc;
}

//
//
CK_RV restore_private_token_object(CK_BYTE * data, CK_ULONG len, OBJECT * pObj)
{
	CK_BYTE *clear = NULL;
	CK_BYTE *obj_data = NULL;
        CK_BYTE *ptr = NULL;
	CK_BYTE *key = NULL;
	CK_ULONG key_len;
	CK_ULONG block_size;
	CK_BYTE hash_sha[SHA1_HASH_SIZE];
	CK_ULONG clear_len, obj_data_len;
	CK_RV rc;

	// format for the object data:
	//    (private flag has already been read at this point)
	//    ---- begin encrypted part
	//       length of object data
	//       object data
	//       SHA of object data
	//    ---- end encrypted part
	//

	clear_len = len;

	clear = (CK_BYTE *) malloc(len);
	if (!clear) {
		TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
		rc = CKR_HOST_MEMORY;
		goto done;
	}

	if ((rc = get_encryption_info(&key_len, &block_size)) != CKR_OK)
		goto done;

	// decrypt the encrypted chunk
	key = malloc(key_len);
	if (!key) {
		rc = ERR_HOST_MEMORY;
		goto done;
	}
	memcpy(key, master_key, key_len);

	rc = decrypt_data_with_clear_key(key, key_len,
			  token_specific.data_store.obj_initial_vector,
			  data, len, clear, &clear_len);
	if (rc != CKR_OK) {
		goto done;
	}

	rc = strip_pkcs_padding(clear, len, &clear_len);

	// if the padding extraction didn't work it means the object was
	// tampered with or the key was incorrect
	//
	if (rc != CKR_OK || (clear_len > len)) {
		TRACE_DEVEL("strip_pkcs_padding failed.\n");
		rc = CKR_FUNCTION_FAILED;
		goto done;
	}

	ptr = clear;

	obj_data_len = *(CK_ULONG_32 *) ptr;

	// prevent buffer overflow in sha_update
	if (obj_data_len > clear_len) {
		TRACE_ERROR("stripped length is greater than clear length\n");
		rc = CKR_FUNCTION_FAILED;
		goto done;
	}

	ptr += sizeof(CK_ULONG_32);
	obj_data = ptr;

	// check the hash
	//
	rc = compute_sha1(ptr, obj_data_len, hash_sha);
	if (rc != CKR_OK) {
		goto done;
	}
	ptr += obj_data_len;

	if (memcmp(ptr, hash_sha, SHA1_HASH_SIZE) != 0) {
		TRACE_ERROR("stored hash does not match restored data hash.\n");
		rc = CKR_FUNCTION_FAILED;
		goto done;
	}
	// okay.  at this point, we're satisfied that nobody has tampered with
	// the token object...
	//

	rc = object_mgr_restore_obj(obj_data, pObj);
	if (rc != CKR_OK) {
		goto done;
	}
	rc = CKR_OK;

done:
	if (clear)
		free(clear);
	if (key)
		free(key);

	return rc;
}

//
//
CK_RV load_masterkey_so(void)
{
	FILE *fp = NULL;
	CK_BYTE hash_sha[SHA1_HASH_SIZE];
	CK_BYTE *cipher = NULL;
	CK_BYTE *clear = NULL;
	CK_BYTE *key = NULL;
	CK_ULONG data_len;
	CK_ULONG cipher_len, clear_len;
	CK_RV rc;
	CK_BYTE fname[PATH_MAX];
	CK_ULONG key_len = 0L;
	CK_ULONG master_key_len = 0L;
	CK_ULONG block_size = 0L;
	char pk_dir_buf[PATH_MAX];

	if ((rc = get_encryption_info_for_clear_key(&key_len,
						    &block_size)) != CKR_OK)
		goto done;

	if ((rc = get_encryption_info(&master_key_len, NULL)) != CKR_OK)
		goto done;

	memset(master_key, 0x0, master_key_len);

	data_len = master_key_len + SHA1_HASH_SIZE;
	clear_len = cipher_len = (data_len + block_size - 1)
		& ~(block_size - 1);

	key = malloc(key_len);
	cipher = malloc(cipher_len);
	clear = malloc(clear_len);
	if (key == NULL || cipher == NULL || clear == NULL) {
		rc = ERR_HOST_MEMORY;
		goto done;
	}

	// this file gets created on C_InitToken so we can assume that it always
	// exists
	//
	sprintf(fname, "%s/MK_SO", get_pk_dir(pk_dir_buf));
	fp = fopen((char *)fname, "r");
	if (!fp) {
		TRACE_ERROR("fopen(%s): %s\n", fname, strerror(errno));
		rc = CKR_FUNCTION_FAILED;
		goto done;
	}
	set_perm(fileno(fp));

	rc = fread(cipher, cipher_len, 1, fp);
	if (rc != 1) {
		TRACE_ERROR("fread() failed.\n");
		rc = CKR_FUNCTION_FAILED;
		goto done;
	}

	// decrypt the master key data using the MD5 of the SO key
	// (we can't use the SHA of the SO key since the SHA of the key is
	// stored in the token data file).
	memcpy(key, so_pin_md5, MD5_HASH_SIZE);
	memcpy(key + MD5_HASH_SIZE, so_pin_md5, key_len - MD5_HASH_SIZE);

	rc = decrypt_data_with_clear_key(key, key_len,
					 token_specific.data_store.pin_initial_vector,
					 cipher, cipher_len,
					 clear, &clear_len);
	if (rc != CKR_OK) {
		TRACE_DEVEL("decrypt_data_with_clear_key failed.\n");
		goto done;
	}

	//
	// technically should strip PKCS padding here but since I already know
	// what the length should be, I don't bother.
	//

	// compare the hashes
	//
	rc = compute_sha1(clear, master_key_len, hash_sha);
	if (rc != CKR_OK) {
		goto done;
	}

	if (memcmp(hash_sha, clear + master_key_len, SHA1_HASH_SIZE) != 0) {
		TRACE_ERROR("masterkey hashes do not match\n");
		rc = CKR_FUNCTION_FAILED;
		goto done;
	}

	memcpy(master_key, clear, master_key_len);
	rc = CKR_OK;

done:
	if (fp)
		fclose(fp);
	if (clear)
		free(clear);
	if (cipher)
		free(cipher);
	if (key)
		free(key);
	return rc;
}

//
//
CK_RV load_masterkey_user(void)
{
	FILE *fp = NULL;
	CK_BYTE hash_sha[SHA1_HASH_SIZE];
	CK_BYTE *cipher = NULL;
	CK_BYTE *clear = NULL;
	CK_BYTE *key = NULL;
	CK_ULONG data_len;
	CK_ULONG cipher_len, clear_len;
	CK_RV rc;
	CK_BYTE fname[PATH_MAX];
	CK_ULONG key_len = 0L;
	CK_ULONG master_key_len = 0L;
	CK_ULONG block_size = 0L;
	char pk_dir_buf[PATH_MAX];

	if ((rc = get_encryption_info_for_clear_key(&key_len,
						    &block_size)) != CKR_OK)
		goto done;

	if ((rc = get_encryption_info(&master_key_len, NULL)) != CKR_OK)
		goto done;

	memset(master_key, 0x0, master_key_len);

	data_len = master_key_len + SHA1_HASH_SIZE;
	clear_len = cipher_len = (data_len + block_size - 1)
				 & ~(block_size - 1);

	key = malloc(key_len);
	cipher = malloc(cipher_len);
	clear = malloc(clear_len);
	if (key == NULL || cipher == NULL || clear == NULL) {
		rc = ERR_HOST_MEMORY;
		goto done;
	}

	// this file gets created on C_InitToken so we can assume that it always
	// exists
	//
	sprintf(fname, "%s/MK_USER", get_pk_dir(pk_dir_buf));
	fp = fopen((char *)fname, "r");
	if (!fp) {
		TRACE_ERROR("fopen(%s): %s\n", fname, strerror(errno));
		rc = CKR_FUNCTION_FAILED;
		goto done;
	}
	set_perm(fileno(fp));

	rc = fread(cipher, cipher_len, 1, fp);
	if (rc != 1) {
		TRACE_ERROR("fread failed.\n");
		rc = CKR_FUNCTION_FAILED;
		goto done;
	}
	// decrypt the master key data using the MD5 of the SO key
	// (we can't use the SHA of the SO key since the SHA of the key is
	// stored in the token data file).
	memcpy(key, user_pin_md5, MD5_HASH_SIZE);
	memcpy(key + MD5_HASH_SIZE, user_pin_md5, key_len - MD5_HASH_SIZE);

	rc = decrypt_data_with_clear_key(key, key_len,
					 token_specific.data_store.pin_initial_vector,
					 cipher, cipher_len,
					 clear, &clear_len);
	if (rc != CKR_OK) {
		TRACE_DEVEL("decrypt_data_with_clear_key failed.\n");
		goto done;
	}

	//
	// technically should strip PKCS padding here but since I already know
	// what the length should be, I don't bother.
	//

	// compare the hashes
	//
	rc = compute_sha1(clear, master_key_len, hash_sha);
	if (rc != CKR_OK) {
		goto done;
	}

	if (memcmp(hash_sha, clear + master_key_len, SHA1_HASH_SIZE) != 0) {
		TRACE_ERROR("User's masterkey hashes do not match.\n");
		rc = CKR_FUNCTION_FAILED;
		goto done;
	}

	memcpy(master_key, clear, master_key_len);
	rc = CKR_OK;

done:
	if (fp)
		fclose(fp);
	if (key)
		free(key);
	if (clear)
		free(clear);
	if (cipher)
		free(cipher);
	return rc;
}

//
//
CK_RV save_masterkey_so(void)
{
	FILE *fp = NULL;
	CK_BYTE *clear = NULL;
	CK_ULONG clear_len = 0L;
	CK_BYTE *cipher = NULL;
	CK_ULONG cipher_len = 0L;
	CK_BYTE *key = NULL;
	CK_ULONG key_len = 0L;
	CK_ULONG master_key_len = 0L;
	CK_ULONG block_size = 0L;
	CK_ULONG data_len = 0L;
	CK_BYTE fname[PATH_MAX];
	CK_RV rc;
	char pk_dir_buf[PATH_MAX];

	/* Skip it if master key is not needed. */
   	if (!token_specific.data_store.use_master_key)
		return CKR_OK;

	if ((rc = get_encryption_info_for_clear_key(&key_len,
						    &block_size)) != CKR_OK)
		goto done;

	if ((rc = get_encryption_info(&master_key_len, NULL)) != CKR_OK)
		goto done;

	data_len = master_key_len + SHA1_HASH_SIZE;
	cipher_len = clear_len = block_size * (data_len / block_size + 1);

	key = malloc(key_len);
	clear = malloc(clear_len);
	cipher = malloc(cipher_len);
	if (key == NULL || clear == NULL || cipher == NULL) {
		TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
		rc = ERR_HOST_MEMORY;
		goto done;
	}

	// Copy data to buffer (key+hash)
	memcpy(clear, master_key, master_key_len);
	if ((rc = compute_sha1(master_key, master_key_len,
			      clear + master_key_len)) != CKR_OK)
		goto done;
	add_pkcs_padding(clear + data_len, block_size, data_len,
			 clear_len);

	// encrypt the key data
	memcpy(key, so_pin_md5, MD5_HASH_SIZE);
	memcpy(key + MD5_HASH_SIZE, so_pin_md5, key_len - MD5_HASH_SIZE);

	rc = encrypt_data_with_clear_key(key, key_len,
					 token_specific.data_store.pin_initial_vector,
					 clear, clear_len,
					 cipher, &cipher_len);
	if (rc != CKR_OK) {
		goto done;
	}

	// write the file
	//
	// probably ought to ensure the permissions are correct
	//
	sprintf(fname, "%s/MK_SO", get_pk_dir(pk_dir_buf));
	fp = fopen((char *)fname, "w");
	if (!fp) {
		TRACE_ERROR("fopen(%s): %s\n", fname, strerror(errno));
		rc = CKR_FUNCTION_FAILED;
		goto done;
	}
	set_perm(fileno(fp));

	rc = fwrite(cipher, cipher_len, 1, fp);
	if (rc != 1) {
		TRACE_ERROR("fwrite failed.\n");
		rc = CKR_FUNCTION_FAILED;
		goto done;
	}

	rc = CKR_OK;

done:
	if (fp)
		fclose(fp);
	if (key)
		free(key);
	if (clear)
		free(clear);
	if (cipher)
		free(cipher);
	return rc;
}

//
//
CK_RV save_masterkey_user(void)
{
	FILE *fp = NULL;
	CK_BYTE *clear = NULL;
	CK_ULONG clear_len = 0L;
	CK_BYTE *cipher = NULL;
	CK_ULONG cipher_len = 0L;
	CK_BYTE *key = NULL;
	CK_ULONG key_len = 0L;
	CK_ULONG master_key_len = 0L;
	CK_ULONG block_size;
	CK_ULONG data_len;
	CK_BYTE fname[PATH_MAX];
	CK_RV rc;
	char pk_dir_buf[PATH_MAX];

	if ((rc = get_encryption_info_for_clear_key(&key_len,
						    &block_size)) != CKR_OK)
		goto done;

	if ((rc = get_encryption_info(&master_key_len, NULL)) != CKR_OK)
		goto done;

	data_len = master_key_len + SHA1_HASH_SIZE;
	cipher_len = clear_len = block_size * (data_len/block_size + 1);

	key = malloc(key_len);
	clear = malloc(clear_len);
	cipher = malloc(cipher_len);
	if (key == NULL || clear == NULL || cipher == NULL) {
		TRACE_ERROR("%s\n", ock_err(ERR_HOST_MEMORY));
		rc = ERR_HOST_MEMORY;
		goto done;
	}

	// Copy data to buffer (key+hash)
	memcpy(clear, master_key, master_key_len);
	if ((rc = compute_sha1(master_key, master_key_len,
			      clear + master_key_len)) != CKR_OK)
		goto done;
	add_pkcs_padding(clear + data_len, block_size , data_len,
			 clear_len);

	// encrypt the key data
	memcpy(key, user_pin_md5, MD5_HASH_SIZE);
	memcpy(key + MD5_HASH_SIZE, user_pin_md5, key_len - MD5_HASH_SIZE);

	rc = encrypt_data_with_clear_key(key, key_len,
					 token_specific.data_store.pin_initial_vector,
					 clear, clear_len,
					 cipher, &cipher_len);
	if (rc != CKR_OK) {
		goto done;
	}

	// write the file
	//
	// probably ought to ensure the permissions are correct
	//
	sprintf(fname, "%s/MK_USER", get_pk_dir(pk_dir_buf));
	fp = fopen((char *)fname, "w");
	if (!fp) {
		TRACE_ERROR("fopen(%s): %s\n", fname, strerror(errno));
		rc = CKR_FUNCTION_FAILED;
		goto done;
	}

	set_perm(fileno(fp));
	rc = fwrite(cipher, cipher_len, 1, fp);
	if (rc != 1) {
		TRACE_ERROR("fwrite failed.\n");
		rc = CKR_FUNCTION_FAILED;
		goto done;
	}

	rc = CKR_OK;

done:
	if (fp)
		fclose(fp);
	if (key)
		free(key);
	if (clear)
		free(clear);
	if (cipher)
		free(cipher);
	return rc;
}

//
//
CK_RV reload_token_object(OBJECT * obj)
{
	FILE *fp = NULL;
	CK_BYTE *buf = NULL;
	CK_BYTE fname[PATH_MAX];
	CK_BBOOL priv;
	CK_ULONG_32 size;
	CK_ULONG size_64;
	CK_RV rc;
	size_t read_size;
	char pk_dir_buf[PATH_MAX];

	memset((char *)fname, 0x0, sizeof(fname));

	sprintf((char *)fname, "%s/%s/", get_pk_dir(pk_dir_buf),
		PK_LITE_OBJ_DIR);

	strncat((char *)fname, (char *)obj->name, 8);

	fp = fopen((char *)fname, "r");
	if (!fp) {
		TRACE_ERROR("fopen(%s): %s\n", fname, strerror(errno));
		rc = CKR_FUNCTION_FAILED;
		goto done;
	}

	set_perm(fileno(fp));

	fread(&size, sizeof(CK_ULONG_32), 1, fp);
	fread(&priv, sizeof(CK_BBOOL), 1, fp);

	size = size - sizeof(CK_ULONG_32) - sizeof(CK_BBOOL);	// SAB

	buf = (CK_BYTE *) malloc(size);
	if (!buf) {
		rc = CKR_HOST_MEMORY;
		OCK_SYSLOG(LOG_ERR,
			   "Cannot malloc %u bytes to read in token object %s "
			   "(ignoring it)", size, fname);
		goto done;
	}

	read_size = fread(buf, 1, size, fp);
	if (read_size != size) {
		OCK_SYSLOG(LOG_ERR,
			   "Token object %s appears corrupted (ignoring it)",
			   fname);
		rc = CKR_FUNCTION_FAILED;
		goto done;
	}

	size_64 = size;

	if (priv)
		rc = restore_private_token_object(buf, size_64, obj);
	else
		rc = object_mgr_restore_obj(buf, obj);

done:
	if (fp)
		fclose(fp);
	if (buf)
		free(buf);
	return rc;
}

extern void set_perm(int);

//
//
CK_RV delete_token_object(OBJECT * obj)
{
	FILE *fp1, *fp2;
	CK_BYTE line[100];
	CK_BYTE objidx[PATH_MAX], idxtmp[PATH_MAX], fname[PATH_MAX];
	char pk_dir_buf[PATH_MAX];

	sprintf((char *)objidx, "%s/%s/%s", get_pk_dir(pk_dir_buf),
		PK_LITE_OBJ_DIR, PK_LITE_OBJ_IDX);
	sprintf((char *)idxtmp, "%s/%s/%s", get_pk_dir(pk_dir_buf),
		PK_LITE_OBJ_DIR, "IDX.TMP");

	// FIXME:  on UNIX, we need to make sure these guys aren't symlinks
	//         before we blindly write to these files...
	//

	// remove the object from the index file
	//

	fp1 = fopen((char *)objidx, "r");
	fp2 = fopen((char *)idxtmp, "w");
	if (!fp1 || !fp2) {
		if (fp1)
			fclose(fp1);
		if (fp2)
			fclose(fp2);
		TRACE_ERROR("fopen failed\n");
		return CKR_FUNCTION_FAILED;
	}

	set_perm(fileno(fp2));

	while (!feof(fp1)) {
		(void)fgets((char *)line, 50, fp1);
		if (!feof(fp1)) {
			line[strlen((char *)line) - 1] = 0;
			if (strcmp((char *)line, (char *)obj->name) == 0)
				continue;
			else
				fprintf(fp2, "%s\n", line);
		}
	}

	fclose(fp1);
	fclose(fp2);
	fp2 = fopen((char *)objidx, "w");
	fp1 = fopen((char *)idxtmp, "r");
	if (!fp1 || !fp2) {
		if (fp1)
			fclose(fp1);
		if (fp2)
			fclose(fp2);
		TRACE_ERROR("fopen failed\n");
		return CKR_FUNCTION_FAILED;
	}

	set_perm(fileno(fp2));

	while (!feof(fp1)) {
		(void)fgets((char *)line, 50, fp1);
		if (!feof(fp1))
			fprintf(fp2, "%s", (char *)line);
	}

	fclose(fp1);
	fclose(fp2);

	sprintf((char *)fname, "%s/%s/%s", get_pk_dir(pk_dir_buf),
		PK_LITE_OBJ_DIR, (char *)obj->name);
	unlink((char *)fname);
	return CKR_OK;

}

CK_RV delete_token_data()
{
	CK_RV rc = CKR_OK;
	char *cmd = NULL;
	char pk_dir_buf[PATH_MAX];

	// Construct a string to delete the token objects.
	//
	// META This should be fine since the open session checking
	// should occur at the API not the STDLL
	//
	// TODO: Implement delete_all_files_in_dir() */
	if (asprintf(&cmd, "%s %s/%s/* > /dev/null 2>&1", DEL_CMD,
			    get_pk_dir(pk_dir_buf), PK_LITE_OBJ_DIR) < 0) {
		rc = CKR_HOST_MEMORY;
		goto done;
	}

	system(cmd);
done:
	free(cmd);
	return rc;
}

CK_RV generate_master_key(CK_BYTE *key)
{
	CK_RV rc = CKR_OK;
	CK_ULONG key_len = 0L;
	CK_ULONG master_key_len = 0L;

	/* Skip it if master key is not needed. */
   	if (!token_specific.data_store.use_master_key)
		return CKR_OK;

	if ((rc = get_encryption_info_for_clear_key(&key_len, NULL)) != CKR_OK ||
	    (rc = get_encryption_info(&master_key_len, NULL)) != CKR_OK)
		return ERR_FUNCTION_FAILED;

	/* For secure key tokens, object encrypt/decrypt uses
	 * software(openssl), not token. So generate masterkey via RNG.
	 */
	if (token_specific.token_keysize)
		return rng_generate(key, key_len);

	/* For clear key tokens, let token generate masterkey
	 * since token will also encrypt/decrypt the objects.
	 */
	switch (token_specific.data_store.encryption_algorithm) {
	case CKM_DES3_CBC:
		return token_specific.t_des_key_gen(key, master_key_len,
						    key_len);
	case CKM_AES_CBC:
		return token_specific.t_aes_key_gen(key, master_key_len,
						    key_len);
	}

	return ERR_MECHANISM_INVALID;
}

void init_data_store(char *directory)
{
	char *pkdir;
	if ((pkdir = getenv("PKCS_APP_STORE")) != NULL) {
		pk_dir = (char *)malloc(strlen(pkdir) + 1024);
		memset(pk_dir, 0, strlen(pkdir) + 1024);
		sprintf(pk_dir, "%s/%s", pkdir, SUB_DIR);
	} else {
		pk_dir = (char *)malloc(strlen(directory) + 25);
		memset(pk_dir, 0, strlen(directory) + 25);
		sprintf(pk_dir, "%s", directory);
	}
}
