/*
 * Author: Andrei Zavada <johnhommer@gmail.com>
 *         building on original work by Thomas Nowotny <tnowotny@ucsd.edu>
 *
 * License: GPL-2+
 *
 * Initial version: 2009-03-31
 *
 */


#ifndef LIBCN_BASE_NEURON_H
#define LIBCN_BASE_NEURON_H

#include <list>
#include <cstring>
#include <cmath>
#include <map>
#include <numeric>

#include "base-unit.hh"
#include "base-synapse.hh"

#include "config.h"



using namespace std;
using namespace CNRun;

namespace CNRun {

#define CN_MIN_DISTANCE .1




class CModel;
struct SSpikeloggerService;


typedef map<C_BaseSynapse*, double> SCleft;
inline double operator+ ( double a, const pair<C_BaseSynapse*, double>& b) { return a + b.second; }

class C_BaseNeuron
  : public C_BaseUnit {

	struct SCoord {
		double x, y, z;

		SCoord( const double& inx, const double& iny, const double& inz)
		      : x (inx), y (iny), z (inz)
			{}

	      // distance
		double operator - ( const SCoord &p)
			{
				return sqrt( pow(x - p.x, 2) + pow(y - p.y, 2) + pow(z - p.z, 2));
			}

		bool operator == ( const SCoord &p) const
			{
				return x == p.x && y == p.y && z == p.z;
			}
		bool operator != ( const SCoord &p) const
			{
				return x != p.x || y != p.y || z != p.z;
			}
		bool too_close( const SCoord& p, double mindist = CN_MIN_DISTANCE) const
			{
				return	fabs(x - p.x) < mindist &&
					fabs(y - p.y) < mindist &&
					fabs(z - p.z) < mindist;
			}
	};

    friend class CModel;
    friend class C_BaseSynapse;

    protected:
	C_BaseNeuron();

	SCleft	_dendrites;
	list<C_BaseSynapse*>
		_axonal_harbour;
    public:
	SCoord	pos;

	size_t axonal_conns() const	{ return _axonal_harbour.size(); }
	size_t dendrites() const	{ return _dendrites.size(); }

	bool connects_to( const C_BaseNeuron &to) const __attribute__ ((pure));
	C_BaseSynapse *connects_via( C_BaseNeuron &to,
				     SCleft::mapped_type *g_ptr = nullptr) const;

    protected:
	C_BaseNeuron( TUnitType intype, const char *inlabel,
		      double inx, double iny, double inz,
		      CModel* inM, int s_mask = 0)
	      : C_BaseUnit (intype, inlabel, inM, s_mask),
		pos (inx, iny, inz),
		_spikelogger_agent (nullptr)
		{}

	virtual ~C_BaseNeuron();

    public:
	void reset_state();

      // even though for rate-based neurons, E is not meaningful
      // leave these here to make the method available to synapses wanting _target-E
	virtual double E() const
		{  return __cn_dummy_double;  }
	virtual double E( vector<double>&) const
		{  return __cn_dummy_double;  }
      // likewise, for those needing _source->F
	virtual double F() const
		{  return __cn_dummy_double;  }
	virtual double F( vector<double>&) const
		{  return __cn_dummy_double;  }

	// struct __SCleft_second_plus {
	// 	double operator() ( double a, const SCleft::value_type &i) { return a + i.second; }
	// };
	double Isyn() const  // is the sum of Isyn() on all dendrites
		{
			double I = 0.;
			for ( auto &Y : _dendrites )
				I += Y.first->Isyn(*this, Y.second);
			return I;
		}

	double Isyn( vector<double> &x) const  // an honourable mention
		{
			double I = 0.;
			for ( auto &Y : _dendrites )
				I += Y.first->Isyn(x, *this, Y.second);
			return I;
		}

	virtual void possibly_fire()
		{}

      // Even though rate-based neurons do not track individual spikes,
      // we can estimate a probability of such a neuron spiking as F*dt*rand().
      // Which makes this method a valid one

      // Note this assumes P[0] is F for all rate-based neurons, and E
      // for those conductance-based, which by now is hard-coded for all neurons.
	virtual unsigned n_spikes_in_last_dt() const
		{  return 0;  }
	virtual void do_detect_spike_or_whatever()
		{}

    protected:
	SSpikeloggerService *_spikelogger_agent;

    public:
	SSpikeloggerService* spikelogger_agent()  { return _spikelogger_agent;  }
	SSpikeloggerService* enable_spikelogging_service( int s_mask = 0);
	SSpikeloggerService* enable_spikelogging_service( double sample_period, double sigma, double from = 0.,
							  int s_mask = 0);
	void disable_spikelogging_service();
	void sync_spikelogging_history();

	double distance_to( C_BaseNeuron*) const; // will do on demand

	void dump( bool with_params = false, FILE *strm = stdout) const;
};





#define CN_KL_COMPUTESDF	(1 << 0)
#define CN_KL_ISSPIKINGNOW	(1 << 1)
#define CN_KL_PERSIST		(1 << 2)  // should not be deleted at disable_spikelogging_service
#define CN_KL_IDLE		(1 << 3)  // should not be placed on spikelogging_neu_list on enable_spikelogging_service


struct SSpikeloggerService {

	friend class C_BaseNeuron;
	friend class C_HostedConductanceBasedNeuron;  // accesses _status from do_spikelogging_or_whatever
	friend class COscillatorDotPoisson;  // same
	friend class COscillatorPoisson;  // same
	friend class CModel;  // checks CN_KL_IDLE in include_unit
    private:
	SSpikeloggerService();

	int _status;

    public:
	SSpikeloggerService( C_BaseNeuron *client,
			     int s_mask = 0)
	      : _status (s_mask & ~CN_KL_COMPUTESDF),
		_client (client),
		t_last_spike_start (-INFINITY), t_last_spike_end (-INFINITY),
		sample_period (42), sigma (42), start_delay (0.)
		{}
	SSpikeloggerService( C_BaseNeuron *client,
			     double insample_period, double insigma, double instart_delay = 0.,
			     int s_mask = 0)
	      : _status (s_mask | CN_KL_COMPUTESDF),
		_client (client),
		t_last_spike_start (-INFINITY), t_last_spike_end (-INFINITY),
		sample_period (insample_period), sigma (insigma), start_delay (instart_delay)
		{}

	C_BaseNeuron *_client;

	double	t_last_spike_start,
		t_last_spike_end;

	size_t n_spikes_since( double since = 0.) const __attribute__ ((pure));

	double	sample_period,
		sigma,
		start_delay;

//	void spike_detect();  // multiplexing units will have a different version
	// replaced by do_spikelogging_or_whatever on the client side

	vector<double> spike_history;

	void reset()
		{
			_status &= ~CN_KL_ISSPIKINGNOW;
			t_last_spike_start = t_last_spike_end
				/*= t_firing_started = t_firing_ended */ = -INFINITY;
			spike_history.clear();
		}

    protected:
	void sync_history();

    public:
      // spike density function
	double sdf( double at, double sample_length, double sigma, unsigned* nspikes = nullptr) const;
      // spike homogeneity function
	double shf( double at, double sample_length) const;

      // why not allow custom sampling?
	size_t get_sxf_vector_custom( vector<double> *sdf_buf, vector<double> *shf_buf, vector<unsigned> *nsp_buf,
			       double sample_period_custom, double sigma_custom,
			       double from = 0., double to = 0.) const; // "to == 0." for model_time()
	size_t get_sxf_vector( vector<double> *sdf_buf, vector<double> *shf_buf, vector<unsigned> *nsp_buf,
			       double from = 0., double to = 0.) const
		{
			return get_sxf_vector_custom( sdf_buf, shf_buf, nsp_buf,
						      sample_period, sigma,
						      from, to);
		}
};




inline void
C_BaseNeuron::reset_state()
{
	C_BaseUnit::reset_state();
	if ( _spikelogger_agent )
		_spikelogger_agent->reset();
}



inline void
C_BaseNeuron::sync_spikelogging_history()
{
	if ( _spikelogger_agent )
		_spikelogger_agent->sync_history();
}



inline double
C_BaseSynapse::g_on_target( const C_BaseNeuron &which) const
{
	return (find( _targets.begin(), _targets.end(), &which) != _targets.end())
		? which._dendrites.at(const_cast<C_BaseSynapse*>(this)) : __cn_dummy_double;
}
inline void
C_BaseSynapse::set_g_on_target( C_BaseNeuron &which, double g)
{
	if ( find( _targets.begin(), _targets.end(), &which) != _targets.end() )
		which._dendrites[this] = g;
}


}

#endif

// EOF
