/*
 *  Copyright 2011 Wolfgang Koller - http://www.gofg.at/
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

function FileError() {
}

FileError.cast = function(p_code) {
    var fe = new FileError();
    fe.code = p_code;

    return fe;
}
FileError.prototype.code = 0;
FileError.NOT_FOUND_ERR = 1;
FileError.SECURITY_ERR = 2;
FileError.ABORT_ERR = 3;
FileError.NOT_READABLE_ERR = 4;
FileError.ENCODING_ERR = 5;
FileError.NO_MODIFICATION_ALLOWED_ERR = 6;
FileError.INVALID_STATE_ERR = 7;
FileError.SYNTAX_ERR = 8;
FileError.INVALID_MODIFICATION_ERR = 9;
FileError.QUOTA_EXCEEDED_ERR = 10;
FileError.TYPE_MISMATCH_ERR = 11;
FileError.PATH_EXISTS_ERR = 12;

function FileException() {
}

FileException.cast = function(p_code) {
    var fe = new FileException();
    fe.code = p_code;

    return fe;
}
FileException.prototype.code = 0;
FileException.NOT_FOUND_ERR = 1;
FileException.SECURITY_ERR = 2;
FileException.ABORT_ERR = 3;
FileException.NOT_READABLE_ERR = 4;
FileException.ENCODING_ERR = 5;
FileException.NO_MODIFICATION_ALLOWED_ERR = 6;
FileException.INVALID_STATE_ERR = 7;
FileException.SYNTAX_ERR = 8;
FileException.INVALID_MODIFICATION_ERR = 9;
FileException.QUOTA_EXCEEDED_ERR = 10;
FileException.TYPE_MISMATCH_ERR = 11;
FileException.PATH_EXISTS_ERR = 12;


function Metadata(p_modificationDate) {
    this.modificationTime = p_modificationDate || null;
    return this;
}

Metadata.cast = function(p_modificationDate) {
    var md = new Metadata(p_modificationDate);
    return md;
}

function Flags(create, exclusive) {
    this.create = create || false;
    this.exclusive = exclusive || false;
};

Flags.cast = function(p_modificationDate) {
    var md = new Metadata(p_modificationDate);
    return md;
}
Flags.cast = function(create, exclusive) {
    var that = new Flags(create, exclusive);
    return that;
};

function Entry() {
}
Entry.prototype.isFile = false;
Entry.prototype.isDirectory = false;
Entry.prototype.name = "";
Entry.prototype.fullPath = "";//fullpath for cordova-test = realFullPath - "/"
Entry.prototype.filesystem = null;
Entry.prototype.getMetadata = function(successCallback, errorCallback) {
    Cordova.exec(successCallback, errorCallback, "com.cordova.File", "getMetadata", [this.fullPath]);
}
Entry.prototype.setMetadata = function(successCallback, errorCallback) {
    //TODO: implement
    //Cordova.exec(successCallback, errorCallback, "com.cordova.File", "setMetadata", [this.fullPath]);
}

Entry.prototype.toURL = function(mimeType) {
    return "file://" + this.fullPath;
}
Entry.prototype.remove = function(successCallback, errorCallback) {
    Cordova.exec(successCallback, errorCallback, "com.cordova.File", "remove", [this.fullPath]);
}
Entry.prototype.getParent = function(successCallback, errorCallback) {
    Cordova.exec(successCallback, errorCallback, "com.cordova.File", "getParent", [this.fullPath]);
}

function File(name, fullPath, type, lastModifiedDate, size) {
    this.name = name || null;
    this.fullPath = fullPath || null;
    this.type = type || null;
    this.lastModifiedDate = lastModifiedDate || null;
    this.size = size || 0;
}

File.cast = function(p_name, p_fullPath, p_type, p_lastModifiedDate, p_size) {
    var f = new File(p_name, p_fullPath, p_type, p_lastModifiedDate, p_size);
    return f;
}

File.prototype.slice = function(start, end) {
    var res = new File(this.name, this.fullPath, this.type, this.lastModifiedDate, this.size);
    res._sliceStart = start;
    res._sliceEnd = end;
    return res;
}

function FileSaver() {
}

FileSaver.createEvent = function(p_type, p_target) {
    var evt = {
        "type": p_type,
        "target": p_target
    };

    return evt;
}

FileSaver.prototype.abort = function() {
    if (this.readyState == FileSaver.INIT || this.readyState == FileSaver.DONE) throw FileException.cast(FileException.INVALID_STATE_ERR);

    this.error = FileError.cast(FileError.ABORT_ERR);
    this.readyState = FileSaver.DONE;

    if (typeof this.onerror === "function") this.onerror(FileSaver.createEvent("error", this));
    if (typeof this.onabort === "function") this.onabort(FileSaver.createEvent("abort", this));
    if (typeof this.onwriteend === "function") this.onwriteend(FileSaver.createEvent("writeend", this));
}

FileSaver.INIT = 0;
FileSaver.WRITING = 1;
FileSaver.DONE = 2;

FileSaver.prototype.readyState = FileSaver.INIT;
FileSaver.prototype.error = new FileError();
FileSaver.prototype.onwritestart = null;
FileSaver.prototype.onprogress = null;
FileSaver.prototype.onwrite = null;
FileSaver.prototype.onabort = null;
FileSaver.prototype.onerror = null;
FileSaver.prototype.onwriteend = null;

function FileWriter(p_file) {
    this.fullPath = p_file.fullPath || "";
    return this;
}

FileWriter.cast = function(p_fullPath, p_length) {
    var tmpFile = new File(null,p_fullPath,null, null,null);
    var fw = new FileWriter(tmpFile);

    return fw;
}

FileWriter.prototype = new FileSaver();
FileWriter.prototype.fullPath = "";
FileWriter.prototype.position = 0;
FileWriter.prototype.length = 0;
FileWriter.prototype._write = function(data, isBinary) {
    // Check if we are able to write
    if (this.readyState === FileSaver.WRITING) throw FileException.cast(FileException.INVALID_STATE_ERR);

    this.readyState = FileSaver.WRITING;
    if (typeof this.onwritestart === "function") this.onwritestart(FileSaver.createEvent("writestart", this));

    var me = this;
    Cordova.exec(function(p_position, p_length) {
        me.position = p_position;
        me.length = p_length;

        me.readyState = FileSaver.DONE;

        if (typeof me.onwrite === "function") me.onwrite(FileSaver.createEvent("write", me));
        if (typeof me.onwriteend === "function") me.onwriteend(FileSaver.createEvent("writeend", me));
    }, function(p_fileError, p_position, p_length) {
        me.position = p_position;
        me.length = p_length;

        me.error = p_fileError;
        me.readyState = FileSaver.DONE;

        if (typeof me.onerror === "function") me.onerror(FileWriter.createEvent("error", me));
        if (typeof me.onwriteend === "function") me.onwriteend(FileWriter.createEvent("writeend", me));
    }, "com.cordova.File", "write", [this.fullPath, this.position, data, isBinary]);
}

FileWriter.prototype.write = function(data) {
    if (data instanceof ArrayBuffer) {
	var binary = "";
	for (var i = 0; i < data.byteLength; i++) {
            binary += String.fromCharCode(data[i]);
	}
        this._write(binary, true);
    } else if ((data instanceof Blob) || (data instanceof File)) {
        var self = this;
        var reader = new FileReader();
        reader.onloadend = function(evt) {
            //FIXME: check for error;
            self._write(evt.target.result, true);
        }
        reader.readAsBinaryString(data);
    } else {
        this._write(data, false);
    }
}

FileWriter.prototype.seek = function(offset) {
    if (this.readyState === FileSaver.WRITING) throw FileException.cast(FileException.INVALID_STATE_ERR);

    if (offset < 0) {
        this.position = Math.max(offset + this.length, 0);
    }
    else if (offset > this.length) {
        this.position = this.length;
    }
    else {
        this.position = offset;
    }
}

FileWriter.prototype.truncate = function(size) {
    // Check if we are able to write
    if (this.readyState == FileSaver.WRITING) throw FileException.cast(FileException.INVALID_STATE_ERR);

    this.readyState = FileSaver.WRITING;
    if (typeof this.onwritestart === "function") this.onwritestart(FileSaver.createEvent("writestart", this));

    var me = this;
    Cordova.exec(function(p_position, p_length) {
        me.position = p_position;
        me.length = p_length;

        me.readyState = FileSaver.DONE;

        if (typeof me.onwrite === "function") me.onwrite(FileSaver.createEvent("write", me));
        if (typeof me.onwriteend === "function") me.onwriteend(FileSaver.createEvent("writeend", me));
    }, function(p_fileError, p_position, p_length) {
        me.position = p_position;
        me.length = p_length;

        me.error = p_fileError;
        me.readyState = FileSaver.DONE;

        if (typeof me.onerror === "function") me.onerror(FileSaver.createEvent("error", me));
        if (typeof me.onwriteend === "function") me.onwriteend(FileSaver.createEvent("writeend", me));
    }, "com.cordova.File", "truncate", [this.fullPath, size]);
}

var originFileReader = window.FileReader;
function forwardRequestToOriginalFileReader(source, methodName, file) {
    var of = new originFileReader();
    for (var i in source) {
        if (source.hasOwnProperty(i))
            of[i] = source[i];
    }
    of[methodName](file);
}

function FileReader() {
}

FileReader.EMPTY = 0;
FileReader.LOADING = 1;
FileReader.DONE = 2;

FileReader.prototype.readyState = FileReader.EMPTY;
FileReader.prototype.result = "";
FileReader.prototype.error = new FileError();
FileReader.prototype.onloadstart = null;
FileReader.prototype.onprogress = null;
FileReader.prototype.onload = null;
FileReader.prototype.onabort = null;
FileReader.prototype.onerror = null;
FileReader.prototype.onloadend = null;

FileReader.prototype.readAsArrayBuffer = function(file) {
    if (file instanceof Blob) {
        forwardRequestToOriginalFileReader(this, 'readAsArrayBuffer', file);
        return;
    }
    function strToArray(str) {
        var res = new Uint8Array(str.length);
        for (var i = 0; i < str.length; i++) {
            res[i] = str.charCodeAt(i);
        }
        return res;
    }

    var origLoadEnd = this.onloadend;
    var origError = this.onerror;
    var self = this;
    function restore() {
        self.onerror = origError;
        self.onloadend = origLoadEnd;
    }
    this.onloadend = function() {
        restore();
        self.result = strToArray(self.result);
        origLoadEnd.apply(this, arguments);
    }
    this.onerror = function() {
        restore();
        origError.apply(this, arguments);
    }
    this.readAsText(file);
};

function createReaderFunc(name) {
    return function(file) {
        if (file instanceof Blob) {
            forwardRequestToOriginalFileReader(this, name, file);
            return;
        }
        this.readyState = FileReader.EMPTY;
        this.result = null;

        this.readyState = FileReader.LOADING;

        if (typeof this.onloadstart === "function") this.onloadstart(FileSaver.createEvent("loadstart", this));

        var me = this;
        var sliceStart = 0, sliceEnd = -1/*-1 is undefined*/, sliced = false;
        if (file._sliceStart !== undefined) {
            sliceStart = file._sliceStart;
            if (file._sliceEnd !== undefined) {
                sliceEnd = file._sliceEnd;
                if (sliceEnd < 0)
                    sliceEnd--;
            }
            sliced = true;
        }
        Cordova.exec(function(p_data) {
            me.readyState = FileReader.DONE;
            me.result = p_data;

            if (typeof me.onload === "function") me.onload(FileSaver.createEvent("load", me));
            if (typeof me.onloadend === "function") me.onloadend(FileSaver.createEvent("loadend", me));
        }, function(p_fileError) {
            me.readyState = FileReader.DONE;
            me.result = null;
            me.error = p_fileError;

            if (typeof me.onloadend === "function") me.onloadend(FileSaver.createEvent("loadend", me));
            if (typeof me.onerror === "function") me.onerror(FileSaver.createEvent("error", me));
        }, "com.cordova.File", name, [file.fullPath, sliced, sliceStart, sliceEnd]);
    };
}

FileReader.prototype.readAsBinaryString = createReaderFunc('readAsBinaryString');
FileReader.prototype.readAsText = createReaderFunc('readAsText');
FileReader.prototype.readAsDataURL = createReaderFunc('readAsDataURL');

FileReader.prototype.abort = function() {
    this.readyState = FileReader.DONE;
    this.result = null;
    this.error = FileError.cast(FileError.ABORT_ERR);

    if (typeof this.onerror === "function") this.onerror(FileSaver.createEvent("error", me)) ;
    if (typeof this.onabort === "function") this.onabort(FileSaver.createEvent("abort", me)) ;
    if (typeof this.onloadend === "function") this.onloadend(FileSaver.createEvent("loadend", me)) ;
}


function FileEntry() {
    this.isFile = true;
    this.isDirectory = false;
}

FileEntry.cast = function(filename, path) {
    var fe = new FileEntry();
    fe.name = filename;
    fe.fullPath = path;

    return fe;
}

FileEntry.prototype = new Entry();
FileEntry.prototype.createWriter = function(successCallback, errorCallback) {
    this.file(function(p_file) {
        successCallback(FileWriter.cast(p_file.fullPath, p_file.size));
    }, errorCallback);
}
FileEntry.prototype.file = function(successCallback, errorCallback) {
    // Lets get the fileinfo
    Cordova.exec(successCallback, errorCallback, "com.cordova.File", "file", [this.fullPath]);
}
FileEntry.prototype.copyTo = function(parent, newName, successCallback, errorCallback) {
    Cordova.exec(successCallback, errorCallback, "com.cordova.File", "copyFile", [this.fullPath, parent.fullPath, newName]);
};

FileEntry.prototype.moveTo = function(parent, newName, successCallback, errorCallback) {
    Cordova.exec(successCallback, errorCallback, "com.cordova.File", "moveFile", [this.fullPath, parent.fullPath, newName]);
};

function DirectoryReader() {
}

DirectoryReader.cast = function(p_fullPath) {
    var dr = new DirectoryReader();
    dr.fullPath = p_fullPath;

    return dr;
}

DirectoryReader.prototype.fullPath = "";
DirectoryReader.prototype.readEntries = function(successCallback, errorCallback) {
    if (this._used) {
        try {
            successCallback([]);
        } catch (e) {}
        return;
    }
    this._used = true;
    Cordova.exec(successCallback, errorCallback, "com.cordova.File", "readEntries", [this.fullPath]);
}

function DirectoryEntry() {
    this.isFile = false;
    this.isDirectory = true;
}

DirectoryEntry.cast = function(dirname, path) {
    var de = new DirectoryEntry();
    de.name = dirname;
    de.fullPath = path;
    return de;
}

DirectoryEntry.prototype = new Entry();
DirectoryEntry.prototype.createReader = function() {
    return DirectoryReader.cast(this.fullPath);
}
DirectoryEntry.prototype.getFile = function(path, options, successCallback, errorCallback) {
    var requestPath = path;

    if (requestPath.charAt(0) !== '/') requestPath = this.fullPath + "/" + requestPath;

    Cordova.exec(successCallback, errorCallback, "com.cordova.File", "getFile", [requestPath, options]);
}
DirectoryEntry.prototype.getDirectory = function(path, options, successCallback, errorCallback) {
    var requestPath = path;

    // Check for a relative path
    if (requestPath.charAt(0) != '/') requestPath = this.fullPath + "/" + requestPath;

    Cordova.exec(successCallback, errorCallback, "com.cordova.File", "getDirectory", [requestPath, options]);
}
DirectoryEntry.prototype.removeRecursively = function(successCallback, errorCallback) {
    Cordova.exec(successCallback, errorCallback, "com.cordova.File", "removeRecursively", [this.fullPath]);
}

DirectoryEntry.prototype.copyTo = function(parent, newName, successCallback, errorCallback) {
    Cordova.exec(successCallback, errorCallback, "com.cordova.File", "copyDir", [this.fullPath, parent.fullPath, newName]);
};

DirectoryEntry.prototype.moveTo = function(parent, newName, successCallback, errorCallback) {
    Cordova.exec(successCallback, errorCallback, "com.cordova.File", "moveDir", [this.fullPath, parent.fullPath, newName]);
};

function FileSystem() {
}

FileSystem.cast = function(fsname, dirname, path) {
    var fs = new FileSystem();
    fs.name = fsname;
    fs.root = DirectoryEntry.cast(dirname, path);

    return fs;
}

FileSystem.prototype.name = "";
FileSystem.prototype.root = null; // Should be a DirectoryEntry

function LocalFileSystem() {
}

LocalFileSystem.TEMPORARY = 0;
LocalFileSystem.PERSISTENT = 1;

LocalFileSystem.prototype.requestFileSystem = function(type, size, successCallback, errorCallback) {
    Cordova.exec(successCallback, errorCallback, "com.cordova.File", "requestFileSystem", [type,size]);
}
LocalFileSystem.prototype.resolveLocalFileSystemURL = function(url, successCallback, errorCallback) {
    Cordova.exec(successCallback, errorCallback, "com.cordova.File", "resolveLocalFileSystemURL", [url]);
}

function FileUploadOptions() {
    this.fileKey = "file";
    this.fileName = "image.jpg";
    this.mimeType = "image/jpeg";
    this.params = {};
    this.chunkedMode = true;
    this.headers = {};
}

function FileUploadResult() {
    this.bytesSent = 0;
    this.responseCode = 400;
    this.response = "";
}

function FileTransferError(code, source, target, http_status) {
    this.code = code;
    this.source = source;
    this.target = target;
    this.http_status = http_status;
}

FileTransferError.FILE_NOT_FOUND_ERR = 1;
FileTransferError.INVALID_URL_ERR = 2;
FileTransferError.CONNECTION_ERR = 3;
FileTransferError.ABORT_ERR = 4;

function FileTransfer() {
    //TODO: implement onprogress
    var self = this;

    this._id = 0;
    this._callQueue = [];
    this._aborted = 0;
    this._callNum = 0;
    Cordova.exec(function (id) {
        Cordova.exec(function(loaded, total) {
            if (typeof(self.onprogress) === 'function') {
                self.onprogress({ lengthComputable: total > 0, total: total, loaded: loaded });
            }
        }, null, "com.cordova.File", "transferRequestSetOnProgress", [ id ]);

        self._id = id;
        for (var i = 0; i < self._callQueue.length; i++) {
            self._callQueue[i]();
        }
        delete self._callQueue;      
    }, null, "com.cordova.File", "newTransferRequest", [  ]);
}

FileTransfer.prototype = {
    upload: function(filePath, server, successCallback, errorCallback, override, trustAllHosts) {
        var self = this;
        if (this._id === 0) {
            var args = arguments;
            this._callQueue.push(function () {
                self.upload.apply(self, args);
            });
            return;
        }

        var callNum = ++this._callNum;

        if (typeof(successCallback) !== "function") return;
        if (typeof(errorCallback) !== "function") errorCallback = new Function();
        if (typeof(override) !== "object") override = {};

        var options = new FileUploadOptions();
        for (var key in options) {
            if (!(options.hasOwnProperty(key) && override.hasOwnProperty(key)))
                continue;
            if (typeof(options[key]) !== typeof(override[key]))
                continue;
            //TODO: check value limits
            options[key] = override[key];
        }
        for (var key in options.params) {
            if (!options.params.hasOwnProperty(key))
                continue;
            options.params[key] = String(options.params[key]);
        }
        for (var key in options.headers) {
            if (!options.headers.hasOwnProperty(key))
                continue;
            if (options.headers[key] instanceof Array) {
                var array = options.headers[key];
                for (var i = 0; i < array.length; i++) {
                    array[i] = String(array[i]);
                }
                options.headers[key] = array.join(', ');
            } else {
                options.headers[key] = String(options.headers[key]);
            }
        }

        var file = new File(null, filePath);
        var reader = new FileReader();
        reader.onerror = function () {
            errorCallback(new FileTransferError(FileTransferError.FILE_NOT_FOUND_ERR, filePath, server, 0));
        };

        reader.readAsDataURL(file);

        reader.onloadend = function () {
            if (!reader.result)
                return;

            if (self._aborted > callNum) {
                errorCallback(new FileTransferError(FileTransferError.ABORT_ERR, filePath, server, 0));
                return;
            }

            var match = reader.result.match(/^data:[^;]*;base64,(.+)$/);
            if (match) {
                //FIXME: stack overflow with large files
                Cordova.exec(function(status, response) {
                    successCallback({ bytesSent: match[1].length, responseCode: status, response: response });
                }, function (status) {
                    if (status === "abort") {
                        errorCallback(new FileTransferError(FileTransferError.ABORT_ERR, filePath, server, 0));
                    } else if (status === "invalidUrl") {
                        errorCallback(new FileTransferError(FileTransferError.INVALID_URL_ERR, filePath, server, 0));
                    } else {
                        errorCallback(new FileTransferError(FileTransferError.CONNECTION_ERR, filePath, server, status));
                    }
                }, "com.cordova.File", "uploadFile", [self._id, server, atob(match[1]), options["fileKey"], options["fileName"], options["mimeType"], options.params, options.headers]);
            }
        }
    },

    download: function(source, target, successCallback, errorCallback, trustAllHosts) {
        var self = this;
        if (this._id === 0) {
            var args = arguments;
            this._callQueue.push(function () {
                self.download.apply(self, args);
            });
            return;
        }

        if (typeof(successCallback) !== "function") return;
        if (typeof(errorCallback) !== "function") errorCallback = new Function();

        ++this._callNum;

        Cordova.exec(function(data) {
            window.requestFileSystem(LocalFileSystem.PERSISTENT, 0, function (fileSystem) {
                fileSystem.root.getFile(target, {create: true, exclusive: false}, function (fileEntry) {
                    fileEntry.createWriter(function (writer) {
                        writer.onwriteend = function () {
                            successCallback(fileEntry);
                        }
                        writer.onerror = errorCallback;

                        writer._write(atob(data), true);
                    }, errorCallback);
                }, function () {
                    errorCallback(new FileTransferError(FileTransferError.FILE_NOT_FOUND_ERR, source, target, 0));
                });
            }, errorCallback);
        }, function (status, body) {
            var error;
            if (status === "abort") {
                error = new FileTransferError(FileTransferError.ABORT_ERR, source, target, 0);
            } else if (status === "invalidUrl") {
                error = new FileTransferError(FileTransferError.INVALID_URL_ERR, source, target, 0);
            } else {
                error = new FileTransferError(FileTransferError.CONNECTION_ERR, source, target, status);
            }
            if (body) {
                error.body = atob(body);
            }

            errorCallback(error);
        }, "com.cordova.File", "downloadFile", [this._id, source]);
    },

    abort: function() {
        if (this._id === 0) {
            var args = arguments;
            var self = this;
            this._callQueue.push(function () {
                self.abort.apply(self, args);
            });
            return;
        }

        this._aborted = ++this._callNum;
        Cordova.exec(null, null, "com.cordova.File", "abortRequests", [this._id]);
    }
};

Cordova.addConstructor("com.cordova.File", function () {
    var localFileSystem = new LocalFileSystem();
    window.requestFileSystem = localFileSystem.requestFileSystem;
    window.resolveLocalFileSystemURI = localFileSystem.resolveLocalFileSystemURL;

    window.FileUploadOptions = FileUploadOptions;
    window.FileTransfer = FileTransfer;
    window.FileTransferError = FileTransferError;
    window.FileError = FileError;
    window.FileReader = FileReader;
    window.FileWriter = FileWriter;
    window.File = File;
    window.LocalFileSystem = LocalFileSystem;
    window.FileEntry = FileEntry;
    window.DirectoryEntry = DirectoryEntry;
    window.DirectoryReader = DirectoryReader;
    window.FileSystem = FileSystem;
    window.FileEntry = FileEntry;
    window.Flags = Flags;
    window.Metadata = Metadata;
    window.FileException = FileException;
});
