"""Helpers for creating servers catered to Juju needs with OpenStack

Specific notes:
* Expects a public address for each machine, as that's what EC2 promises,
  would be good to weaken this requirement.
* Creates a per-machine security group in case need to poke ports open later,
  as EC2 doesn't support changing groups later, but OpenStack does.
* Needs to tell cloud-init how to get server id, currently in essex metadata
  service gives i-08x style only, so cheat and use filestorage.
* Config must specify an image id, as there's no standard way of looking up
  from distro series across clouds yet in essex.
* Would be really nice to put the service name in the server name, but it's
  not passed down into LaunchMachine currently.

There are some race issues with the current setup:
* Storing of server id needs to complete before cloud-init does the lookup,
  extremely unlikely (two successive api calls vs instance boot and running).
* for environments configured with floating-ips. A floating ip may be
  assigned to another server before the current one finishes launching and
  can use the available ip itself.
"""

from cStringIO import StringIO

from twisted.internet.defer import inlineCallbacks, returnValue


from juju.errors import ProviderError, ProviderInteractionError
from juju.lib import twistutils
from juju.providers.common.launch import LaunchMachine
from juju.providers.common.instance_type import TypeSolver, InstanceType

from .machine import machine_from_instance, get_server_status

from .client import log


class NovaLaunchMachine(LaunchMachine):
    """OpenStack Nova operation for creating a server"""

    _DELAY_FOR_ADDRESSES = 5  # seconds

    @inlineCallbacks
    def start_machine(self, machine_id, zookeepers):
        """Actually launch an instance on Nova.

        :param str machine_id: the juju machine ID to assign

        :param zookeepers: the machines currently running zookeeper, to which
            the new machine will need to connect
        :type zookeepers: list of
            :class:`juju.providers.openstack.machine.NovaProviderMachine`

        :return: a singe-entry list containing a
            :class:`juju.providers.openstack.machine.NovaProviderMachine`
            representing the newly-launched machine
        :rtype: :class:`twisted.internet.defer.Deferred`
        """
        cloud_init = self._create_cloud_init(machine_id, zookeepers)
        cloud_init.set_provider_type(self._provider.provider_type)
        filestorage = self._provider.get_file_storage()

        # Only the master is required to get its own instance id like this.
        if self._master:
            id_name = "juju_master_id"
            # If Swift does not have a valid certificate, by default curl will
            # print a complaint to stderr and nothing to stdout. This makes
            # cloud-init think the instance id is an empty string. Work around
            # by allowing insecure connections if https certs are unchecked.
            if self._provider._check_certs:
                curl = "curl"
            else:
                curl = "curl -k"
            cloud_init.set_instance_id_accessor("$(%s %s)" % (
                curl, filestorage.get_url(id_name),))
        user_data = cloud_init.render()

        # For openstack deployments, really need image id configured as there
        # are no standards to provide a fallback value.
        image_id = self._provider.config.get("default-image-id")
        if image_id is None:
            raise ProviderError("Need to specify a default-image-id")
        security_groups = (
            yield self._provider.port_manager.ensure_groups(machine_id))

        # Find appropriate instance type for the given constraints. Warn
        # if deprecated is instance-type is being used.
        flavor_name = self._provider.config.get("default-instance-type")
        if flavor_name is not None:
            log.warning(
                "default-instance-type is deprecated, use cli --constraints")
        flavors = yield self._provider.nova.list_flavor_details()
        flavor_id = _solve_flavor(self._constraints, flavor_name, flavors)

        hints = self._constraints["os-scheduler-hints"]

        server = yield self._provider.nova.run_server(
            name="juju %s instance %s" %
                (self._provider.environment_name, machine_id,),
            image_id=image_id,
            flavor_id=flavor_id,
            security_group_names=security_groups,
            user_data=user_data,
            scheduler_hints=hints,
            )

        if self._master:
            yield filestorage.put(id_name, StringIO(str(server['id'])))

        # For private clouds allow an option of attaching public
        # floating ips to all the machines. None of the extant public
        # clouds need this.
        if self._provider.config.get('use-floating-ip'):
            # Not possible to attach a floating ip to a newly booted
            # server, must wait for networking to be ready when some
            # kind of address exists.
            while not server.get('addresses'):
                status = get_server_status(server)
                if status != "pending":
                    raise ProviderInteractionError(
                        "Server out of pending status "
                        "without addresses set: %r" % server)
                # Bad, bad place to be doing a wait loop directly
                yield twistutils.sleep(self._DELAY_FOR_ADDRESSES)
                log.debug("Waited for %d seconds for networking on server %r",
                          self._DELAY_FOR_ADDRESSES, server['id'])
                server = yield self._provider.nova.get_server(server['id'])
            yield _assign_floating_ip(self._provider, server['id'])

        returnValue([machine_from_instance(server)])


def _solve_flavor(constraints, flavor_name, flavors):
    flavor_map, flavor_types = {}, {}
    for f in flavors:
        flavor_map[f['name']] = f['id']
        # Arch needs to be part of nova details.
        flavor_types[f['name']] = InstanceType(
            'amd64', f['vcpus'], f['ram'])

    solver = TypeSolver(flavor_types)
    flavor_name = solver.run(constraints)
    if not flavor_name in flavor_map:
        ProviderError("Unknown instance type given: %r" % (flavor_name,))
    return flavor_map[flavor_name]


@inlineCallbacks
def _assign_floating_ip(provider, server_id):
    floating_ips = yield provider.nova.list_floating_ips()
    for floating_ip in floating_ips:
        if floating_ip['instance_id'] is None:
            break
    else:
        floating_ip = yield provider.nova.allocate_floating_ip()
    yield provider.nova.add_floating_ip(server_id, floating_ip['ip'])
