/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.netbeans.modules.options.keymap;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.AbstractListModel;
import javax.swing.JFileChooser;
import javax.swing.KeyStroke;
import org.netbeans.core.options.keymap.api.KeyStrokeUtils;
import org.netbeans.core.options.keymap.api.ShortcutAction;
import org.openide.DialogDisplayer;
import org.openide.NotifyDescriptor;
import org.openide.NotifyDescriptor.InputLine;
import org.openide.NotifyDescriptor.Message;
import org.openide.util.Exceptions;
import org.openide.util.NbBundle;
import org.openide.util.Utilities;
import org.openide.xml.EntityCatalog;
import org.openide.xml.XMLUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Panel for managing keymap profiles
 * @author Max Sauer
 */
public class ProfilesPanel extends javax.swing.JPanel {
    private static final Logger LOG = Logger.getLogger(ProfilesPanel.class.getName());
    
    private ProfileListModel model;
    private KeymapPanel keymapPanel;

    /** Creates new form ProfilesPanel */
    public ProfilesPanel(KeymapPanel k) {
        keymapPanel = k;
        model = new ProfileListModel();
        initComponents();
        model.setData(getKeymapPanel().getMutableModel().getProfiles());
        profilesList.setSelectedValue(getKeymapPanel().getMutableModel().getCurrentProfile(), true);
    }

    private KeymapPanel getKeymapPanel() {
        return keymapPanel;
    }
    
    ProfileListModel getModel() {
        return model;
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        jScrollPane1 = new javax.swing.JScrollPane();
        profilesList = new javax.swing.JList();
        duplicateButton = new javax.swing.JButton();
        restoreButton = new javax.swing.JButton();
        deleteButton = new javax.swing.JButton();
        exportButton = new javax.swing.JButton();
        importButton = new javax.swing.JButton();

        profilesList.setModel(model);
        profilesList.setSelectionMode(javax.swing.ListSelectionModel.SINGLE_SELECTION);
        profilesList.addListSelectionListener(new javax.swing.event.ListSelectionListener() {
            public void valueChanged(javax.swing.event.ListSelectionEvent evt) {
                profilesListValueChanged(evt);
            }
        });
        jScrollPane1.setViewportView(profilesList);
        profilesList.getAccessibleContext().setAccessibleName(org.openide.util.NbBundle.getMessage(ProfilesPanel.class, "ProfilesPanel.profilesList.AccessibleContext.accessibleName")); // NOI18N
        profilesList.getAccessibleContext().setAccessibleDescription(org.openide.util.NbBundle.getMessage(ProfilesPanel.class, "ProfilesPanel.profilesList.AccessibleContext.accessibleDescription")); // NOI18N

        org.openide.awt.Mnemonics.setLocalizedText(duplicateButton, org.openide.util.NbBundle.getMessage(ProfilesPanel.class, "ProfilesPanel.duplicateButton.text")); // NOI18N
        duplicateButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                duplicateButtonActionPerformed(evt);
            }
        });

        org.openide.awt.Mnemonics.setLocalizedText(restoreButton, org.openide.util.NbBundle.getMessage(ProfilesPanel.class, "ProfilesPanel.restoreButton.text")); // NOI18N
        restoreButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                restoreButtonActionPerformed(evt);
            }
        });

        org.openide.awt.Mnemonics.setLocalizedText(deleteButton, org.openide.util.NbBundle.getMessage(ProfilesPanel.class, "ProfilesPanel.deleteButton.text")); // NOI18N
        deleteButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                deleteButtonActionPerformed(evt);
            }
        });

        org.openide.awt.Mnemonics.setLocalizedText(exportButton, org.openide.util.NbBundle.getMessage(ProfilesPanel.class, "ProfilesPanel.exportButton.text")); // NOI18N
        exportButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                exportButtonActionPerformed(evt);
            }
        });

        org.openide.awt.Mnemonics.setLocalizedText(importButton, org.openide.util.NbBundle.getMessage(ProfilesPanel.class, "ProfilesPanel.importButton.text")); // NOI18N
        importButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                importButtonActionPerformed(evt);
            }
        });

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(jScrollPane1, javax.swing.GroupLayout.DEFAULT_SIZE, 234, Short.MAX_VALUE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(duplicateButton, javax.swing.GroupLayout.Alignment.TRAILING)
                    .addComponent(restoreButton, javax.swing.GroupLayout.Alignment.TRAILING)
                    .addComponent(deleteButton, javax.swing.GroupLayout.Alignment.TRAILING)
                    .addComponent(exportButton, javax.swing.GroupLayout.Alignment.TRAILING)
                    .addComponent(importButton, javax.swing.GroupLayout.Alignment.TRAILING))
                .addContainerGap())
        );

        layout.linkSize(javax.swing.SwingConstants.HORIZONTAL, new java.awt.Component[] {deleteButton, duplicateButton, exportButton, importButton, restoreButton});

        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING, false)
                    .addComponent(jScrollPane1, javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(javax.swing.GroupLayout.Alignment.LEADING, layout.createSequentialGroup()
                        .addComponent(duplicateButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(restoreButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(deleteButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                        .addComponent(exportButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(importButton)))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        jScrollPane1.getAccessibleContext().setAccessibleName(org.openide.util.NbBundle.getMessage(ProfilesPanel.class, "ProfilesPanel.jScrollPane1.AccessibleContext.accessibleName")); // NOI18N
        jScrollPane1.getAccessibleContext().setAccessibleDescription(org.openide.util.NbBundle.getMessage(ProfilesPanel.class, "ProfilesPanel.jScrollPane1.AccessibleContext.accessibleDescription")); // NOI18N
        duplicateButton.getAccessibleContext().setAccessibleDescription(org.openide.util.NbBundle.getMessage(ProfilesPanel.class, "ProfilesPanel.duplicateButton.AccessibleContext.accessibleDescription")); // NOI18N
        restoreButton.getAccessibleContext().setAccessibleDescription(org.openide.util.NbBundle.getMessage(ProfilesPanel.class, "ProfilesPanel.restoreButton.AccessibleContext.accessibleDescription")); // NOI18N
        deleteButton.getAccessibleContext().setAccessibleDescription(org.openide.util.NbBundle.getMessage(ProfilesPanel.class, "ProfilesPanel.deleteButton.AccessibleContext.accessibleDescription")); // NOI18N
        exportButton.getAccessibleContext().setAccessibleDescription(org.openide.util.NbBundle.getMessage(ProfilesPanel.class, "ProfilesPanel.exportButton.AccessibleContext.accessibleDescription")); // NOI18N
        importButton.getAccessibleContext().setAccessibleDescription(org.openide.util.NbBundle.getMessage(ProfilesPanel.class, "ProfilesPanel.importButton.AccessibleContext.accessibleDescription")); // NOI18N

        getAccessibleContext().setAccessibleName(org.openide.util.NbBundle.getMessage(ProfilesPanel.class, "ProfilesPanel.AccessibleContext.accessibleName")); // NOI18N
        getAccessibleContext().setAccessibleDescription(org.openide.util.NbBundle.getMessage(ProfilesPanel.class, "ProfilesPanel.AccessibleContext.accessibleDescription")); // NOI18N
    }// </editor-fold>//GEN-END:initComponents

    private void restoreButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_restoreButtonActionPerformed
        deleteOrRestoreSelectedProfile();
    }//GEN-LAST:event_restoreButtonActionPerformed

    private void deleteButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_deleteButtonActionPerformed
        deleteOrRestoreSelectedProfile();
    }//GEN-LAST:event_deleteButtonActionPerformed

    private void profilesListValueChanged(javax.swing.event.ListSelectionEvent evt) {//GEN-FIRST:event_profilesListValueChanged
        String profile = (String) profilesList.getSelectedValue();
        if (profile == null) {
            deleteButton.setEnabled(false);
            duplicateButton.setEnabled(false);
            exportButton.setEnabled(false);
            restoreButton.setEnabled(false);
            return ;
        }
        
        duplicateButton.setEnabled(true);
        exportButton.setEnabled(true);
        
        MutableShortcutsModel model = getKeymapPanel().getMutableModel();
        if(model.isCustomProfile(profile)) {
            deleteButton.setEnabled(true);
            restoreButton.setEnabled(false);
        } else {
            boolean change = model.isChangedProfile(profile);
            if (!change) {
                change |= model.differsFromDefault(profile);
            }
            deleteButton.setEnabled(false);
            restoreButton.setEnabled(change);
        }
    }//GEN-LAST:event_profilesListValueChanged

    private void duplicateButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_duplicateButtonActionPerformed
        duplicateProfile();
    }//GEN-LAST:event_duplicateButtonActionPerformed

    private String duplicateProfile() {
        String newName = null;
        InputLine il = new InputLine(
                KeymapPanel.loc("CTL_Create_New_Profile_Message"), // NOI18N
                KeymapPanel.loc("CTL_Create_New_Profile_Title") // NOI18N
                );
        String profileToDuplicate =(String) profilesList.getSelectedValue();
        il.setInputText(profileToDuplicate);
        DialogDisplayer.getDefault().notify(il);
        if (il.getValue() == NotifyDescriptor.OK_OPTION) {
            newName = il.getInputText();
            for (String s : getKeymapPanel().getMutableModel().getProfiles()) {
                if (newName.equals(s)) {
                    Message md = new Message(
                            KeymapPanel.loc("CTL_Duplicate_Profile_Name"), // NOI18N
                            Message.ERROR_MESSAGE);
                    DialogDisplayer.getDefault().notify(md);
                    return null;
                }
            }
            MutableShortcutsModel currentModel = getKeymapPanel().getMutableModel();
            String currrentProfile = currentModel.getCurrentProfile();
            getKeymapPanel().getMutableModel().setCurrentProfile(profileToDuplicate);
            getKeymapPanel().getMutableModel().cloneProfile(newName);
            currentModel.setCurrentProfile(currrentProfile);
            model.addItem(newName);
            profilesList.setSelectedValue(il.getInputText(), true);
        }
        return newName;
    }

    private static final String PUBLIC_ID = "-//NetBeans//DTD Keymap Preferences 1.0//EN"; //NOI18N
    private static final String SYSTEM_ID = "http://www.netbeans.org/dtds/KeymapPreferences-1_0.dtd"; //NOI18N
    private static final String ATTR_ACTION_ID = "id"; //NOI18N
    private static final String ELEM_SHORTCUT = "shortcut"; //NOI18N
    private static final String ELEM_XML_ROOT = "keymap-preferences"; //NOI18N
    private static final String ATTR_SHORTCUT_STRING = "shortcut_string"; //NOI18N
    private static final String ELEM_ACTION = "action"; //NOI18N

    private void exportButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_exportButtonActionPerformed

        // show filechooser
        final JFileChooser chooser = getFileChooser();

        chooser.setSelectedFile(new File("exported-" + profilesList.getSelectedValue() + "-profile.xml")); //NOI18N

        int ret = chooser.showSaveDialog(this);
        if (ret == JFileChooser.APPROVE_OPTION) {
            Document doc = XMLUtil.createDocument(ELEM_XML_ROOT, null, PUBLIC_ID, SYSTEM_ID);
            Node root = doc.getElementsByTagName(ELEM_XML_ROOT).item(0);

            MutableShortcutsModel kmodel = getKeymapPanel().getMutableModel();
            for (Object o : kmodel.getItems("")) {
                ShortcutAction sca = (ShortcutAction) o;
                String[] shortcuts = kmodel.getShortcuts(sca);
                if (shortcuts.length > 0) { //export only actions with at least one SC
                    String id = sca.getId();
                    Element actionElement = doc.createElement(ELEM_ACTION);
                    actionElement.setAttribute(ATTR_ACTION_ID, id);
                    for (int i = 0; i < shortcuts.length; i++) {
                        Element shortcutElement = doc.createElement(ELEM_SHORTCUT);
                        //get portable representation of the ahortcut
                        String shortcutToStore = shortcutToPortableRepresentation(shortcuts[i]);
                        shortcutElement.setAttribute(ATTR_SHORTCUT_STRING, shortcutToStore);
                        actionElement.appendChild(shortcutElement);
                    }
                    root.appendChild(actionElement);
                }
            }
            File f = chooser.getSelectedFile();

            FileOutputStream fos;
            try {
                fos = new FileOutputStream(f);
                XMLUtil.write(doc, fos, "UTF-8"); //NOI18N
                fos.close();
            } catch (IOException ex) {
                NotifyDescriptor nd = new NotifyDescriptor(
                        NbBundle.getMessage(ProfilesPanel.class, "Export.io.error", ex.getLocalizedMessage()), 
                        NbBundle.getMessage(ProfilesPanel.class, "Export.failed.title"), 
                        NotifyDescriptor.DEFAULT_OPTION, 
                        NotifyDescriptor.INFORMATION_MESSAGE, new Object[] { NotifyDescriptor.OK_OPTION }, NotifyDescriptor.OK_OPTION);
                LOG.log(Level.INFO, "Failed to export bindings", ex);
                DialogDisplayer.getDefault().notify(nd);
            }
        }

    }//GEN-LAST:event_exportButtonActionPerformed

    private static String shortcutToPortableRepresentation(String key) {
        assert key != null : "The parameter key must not be null"; //NOI18N

        StringBuilder buf = new StringBuilder();
        String delimiter = " "; //NOI18N

        for(StringTokenizer st = new StringTokenizer(key, delimiter); st.hasMoreTokens();) { //NOI18N
            String ks = st.nextToken().trim();

            KeyStroke keyStroke = KeyStrokeUtils.getKeyStroke(ks);

            if (keyStroke != null) {
                buf.append(Utilities.keyToString(keyStroke, true));
                if (st.hasMoreTokens())
                    buf.append(' ');
            } else {
                return null;
            }
        }

        return buf.toString();
    }

    private static JFileChooser getFileChooser() {
        final JFileChooser chooser = new JFileChooser();
        XMLFileFilter filter = new XMLFileFilter();

        chooser.setAcceptAllFileFilterUsed(false);
        chooser.addChoosableFileFilter(filter);
        chooser.setFileFilter(filter);

        return chooser;
    }

    private void importButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_importButtonActionPerformed
        JFileChooser chooser = getFileChooser();
        int ret = chooser.showOpenDialog(this);
        boolean notFound = false;
        Throwable err = null;
        
        if(ret != JFileChooser.APPROVE_OPTION) {
            return;
        }
        MutableShortcutsModel kmodel = getKeymapPanel().getMutableModel();
        String newProfile = null;
        try {
            InputSource is = new InputSource(new FileInputStream(chooser.getSelectedFile()));
            newProfile = duplicateProfile();
            if (newProfile == null) return; //invalid (duplicate) profile name
            kmodel.setCurrentProfile(newProfile);

            Document doc = XMLUtil.parse(is, false, true, null, EntityCatalog.getDefault());
            Node root = doc.getElementsByTagName(ELEM_XML_ROOT).item(0);
            if (root == null) {
                throw new IOException(NbBundle.getMessage(ProfilesPanel.class, "Import.invalid.file"));
            }
            NodeList nl = root.getChildNodes();
            for (int i = 0; i < nl.getLength(); i++) {//iterate stored actions
                Node action = nl.item(i);
                final NamedNodeMap attributes = action.getAttributes();
                if (attributes == null) continue;
                String id = attributes.item(0).getNodeValue();
                ShortcutAction sca = kmodel.findActionForId(id);
                NodeList childList = action.getChildNodes();
                int childCount = childList.getLength();
                Set<String> shortcuts = new LinkedHashSet<String>(childCount);
                for (int j = 0; j < childCount; j++) {//iterate shortcuts
                    //iterate shortcuts
                    NamedNodeMap attrs = childList.item(j).getAttributes();
                    if (attrs != null) {
                        String sc = attrs.item(0).getNodeValue();
                        shortcuts.add(ExportShortcutsAction.portableRepresentationToShortcut(sc));
                    }
                }
                if (sca == null) {
                    notFound = true;
                    LOG.log(Level.WARNING, "Failed to import binding for: {0}, keys: {1}", new Object[] { id, shortcuts });
                    continue;
                } else {
                    kmodel.setShortcuts(sca, shortcuts);
                }
            }

        } catch (IOException | SAXException ex) {
            err = ex;
        }
        
        String msg = null;
        
        if (err != null) {
            msg = NbBundle.getMessage(ProfilesPanel.class, "Import.io.error", err.getLocalizedMessage());
            // attempt to remove the newly created profile:
            if (newProfile != null) {
                kmodel.deleteOrRestoreProfile(newProfile);
                model.removeItem(newProfile);
                profilesList.clearSelection();
            }
        } else if (notFound) {
            msg = NbBundle.getMessage(ProfilesPanel.class, "Import.failed.unknown.id");
        }
        
        if (msg != null) {
            NotifyDescriptor nd = new NotifyDescriptor(
                    msg, 
                    NbBundle.getMessage(ProfilesPanel.class, "Import.failed.title"), 
                    NotifyDescriptor.DEFAULT_OPTION, 
                    NotifyDescriptor.INFORMATION_MESSAGE, new Object[] { NotifyDescriptor.OK_OPTION }, NotifyDescriptor.OK_OPTION);
        
            DialogDisplayer.getDefault().notify(nd);
        }

    }//GEN-LAST:event_importButtonActionPerformed

    private static class XMLFileFilter extends javax.swing.filechooser.FileFilter {

        public boolean accept(File file) {            
            if (file.isDirectory()) 
                return true;
            
            if(file.getAbsolutePath().endsWith(".xml")) {
                return true;
            } else {
                return false;
            }
        }
        
        @Override
        public String getDescription() {
            return "XML " + NbBundle.getMessage(ProfilesPanel.class, "CTL_Files") + "(*.xml)"; //NOI18N
        }
    }


    private void deleteOrRestoreSelectedProfile() {
        String currentProfile = (String) profilesList.getSelectedValue();
        final MutableShortcutsModel keymapModel = getKeymapPanel().getMutableModel();
        if (keymapModel.deleteOrRestoreProfile(currentProfile)) {
            model.removeItem(profilesList.getSelectedIndex());
            profilesList.setSelectedIndex(0);
        } else {
            profilesListValueChanged(null);
        }
    }

    public String getSelectedProfile() {
        return (String) profilesList.getSelectedValue();
    }

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton deleteButton;
    private javax.swing.JButton duplicateButton;
    private javax.swing.JButton exportButton;
    private javax.swing.JButton importButton;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JList profilesList;
    private javax.swing.JButton restoreButton;
    // End of variables declaration//GEN-END:variables


    private class ProfileListModel extends AbstractListModel {

        private ArrayList<String> delegate = new ArrayList<String>();

        public int getSize() {
            return delegate.size();
        }

        public Object getElementAt(int index) {
            return delegate.get(index);
        }

        public void setData(Collection<String> c) {
            delegate.clear();
            delegate.addAll(c);
            fireContentsChanged(this, 0, c.size());
        }

        private void addItem(String inputText) {
            delegate.add(inputText);
            int size = delegate.size();
            fireContentsChanged(this, size, size);
        }

        private void removeItem(int index) {
            delegate.remove(index);
            fireContentsChanged(this, index, index);
        }
        
        private boolean removeItem(String id) {
            int idx = delegate.indexOf(id);
            boolean b = delegate.remove(id);
            if (b) {
                fireContentsChanged(this, idx, idx);
            }
            return b;
        }
    }

}
