/* regexp.hh - regular expressions
 * Copyright 2004 Bas Wijnen <wijnen@debian.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef SHEVEK_REGEXP_HH
#define SHEVEK_REGEXP_HH

#include <glibmm.h>
#include <sys/types.h>
#include <regex.h>
#include "debug.hh"

namespace shevek
{
  /// Use regular expressions.
  class regexp
  {
    regex_t m_buffer;
    bool m_case_sensitive;
    regmatch_t *m_match;
    unsigned m_size;
    std::string m_pattern, m_data;
    void l_setup (std::string const &pattern, bool destroy);
  public:
    /// Create a new object, and optionally fill it with a pattern.
    regexp (std::string const &pattern = std::string (),
	    bool case_sensitive = false);
    /// Set a pattern, removing the previous one.
    regexp &operator= (std::string const &pattern);
    /// Copy a regexp.
    regexp (regexp const &that);
    /// Copy a regexp.
    regexp &operator= (regexp const &that);
    /// Set whether the evaluation should be case sensitive.
    void case_sensitive (bool value = true);
    /// Destructor, this cleans up internal structures.
    ~regexp ();
    /// Check whether the pattern matches a string, and fill internal match structures if it does.
    bool operator() (std::string const &data);
    /// Retrieve the value of a subexpression from the last matched string.
    /** This throws an exception if the subexpression is not valid.
     */
    std::string operator[] (unsigned idx) const;
    /// Test whether a subexpression was filled by the last matching string.
    bool valid (unsigned idx) const;
    /// Get the number of subexpression.
    unsigned size () const;
    /// Transform a string with \-codes according to the last matching string.
    std::string transform (std::string const &data) const;
    /// Get the current pattern.
    std::string const &pattern () const;
  };
}

#endif
