/*
 * NAT - An universal Translator
 * Copyright (C) 2005 Bruno Mascret
 * Contact: bmascret@free.fr
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package outils;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.util.ArrayList;

import nat.ConfigNat;
/**
 * Classe permettant de mettre en place et de configurer la coupure littéraire
 * à partir de dictionnaires de patterns de coupure (basé sur l'algo de Liang)
 * @author bruno
 *
 */
public class HyphenationToolkit
{
	/** Encodage par défaut de la feuille xsl de coupure */
	private static final String encodageDefaut= "UTF-8";
	/**
	 * Construit le dictionnaire de coupure (en xsl) à partir d'un dictionnaire existant 
	 * au format .dic (Tex)
	 * <p>Adapte les patterns pour les expressions régulières en xsl:</p>
	 * <ul>
	 * 	<li>remplacement des . du début par ^</li>
	 * 	<li>remplacement des . de fin par $</li>
	 * 	<li>doublement des apostrophes</li>
	 * </ul>
	 * <p>Fabrique le fichier hyphenation.xsl contenant les patterns de coupure</p>
	 * @param dico l'adresse du dictionnaire existant au format .dic
	 * @param dicoNat adresse de la feuille xsl de coupure
	 * @param encodage encodage du dictionnaire <code>dico</code>
	 * @return true si le dictionnaire a été créé correctement, false sinon
	 * @since 2.0
	 */
	public static boolean fabriqueDico(String dico, String dicoNat,String encodage)
	{
		boolean retour = true;
		if(encodage==null){encodage="UTF-8";}
		BufferedWriter bw = null;
		ArrayList<String> listes = new ArrayList<String>();
		//lecture et stockage des patterns dans une ArrayList
		boolean fini=false;
		try
		{
			BufferedReader br = new BufferedReader(new InputStreamReader(new FileInputStream(dico),encodage));
			//première ligne donnant encodage ?
			String l1 = br.readLine();
			if(!l1.toLowerCase().equals(l1))//c'est pas une règle a priori
			{
				br = new BufferedReader(new InputStreamReader(new FileInputStream(dico),l1));
				br.readLine();//on lit la 1ère ligne de nouveau pour la virer...
			}
			else{br = new BufferedReader(new InputStreamReader(new FileInputStream(dico),encodage));}
			//lecture 
			while(!fini)
			{
				String lu = br.readLine();
				if(lu==null){fini=true;}
				else
				{
					if(lu.startsWith(".")){lu = "^" + lu.substring(1);}
					if(lu.endsWith(".")){lu = lu.substring(0, lu.lastIndexOf("."))+"$";}
					lu = lu.replace("'", "''");
					listes.add(lu);
				}
			}
			br.close();
			//écriture du fichier xsl contenant le paramètre patterns d'après la liste précédemment chargée
			bw=new BufferedWriter(new OutputStreamWriter(new FileOutputStream(dicoNat),encodage));
			//entêtes
			bw.write("<xsl:stylesheet version=\"2.0\"\n"+
				"xmlns:xsl='http://www.w3.org/1999/XSL/Transform'\n"+ 
				"xmlns:xs='http://www.w3.org/2001/XMLSchema'>\n"+
				"<!-- auto-generated by NAT from "+ dico +"-->\n");
			//les patterns
			bw.write("<xsl:param name=\"patterns\" as=\"xs:string*\" select=\"tokenize('");
			bw.write(listes.get(0));
			for (int i=1;i<listes.size();i++)
			{
				bw.write("," + listes.get(i));				
			}
			bw.write("',',')\"/>");
			//la fin de la feuille
			bw.write("\n</xsl:stylesheet>");
			bw.close();
		}
		catch(FileNotFoundException fnfe){fnfe.printStackTrace();retour=false;}
		catch (IOException ioe){ioe.printStackTrace();retour=false;}
		return retour;
	}
	/**
	 * Construit le dictionnaire de coupure de nat (en xsl) à partir d'un dictionnaire existant 
	 * au format .dic (Tex)
	 * <p>Adapte les patterns pour les expressions régulières en xsl:</p>
	 * <ul>
	 * 	<li>remplacement des . du début par ^</li>
	 * 	<li>remplacement des . de fin par $</li>
	 * 	<li>doublement des apostrophes</li>
	 * </ul>
	 * <p>Fabrique le fichier hyphen.xsl contenant les patterns de coupure</p>
	 * @param dico l'adresse du dictionnaire existant au format .dic
	 * @param dicoNat adresse de la feuille xsl de coupure
	 * @param encodage encodage du dictionnaire <code>dico</code>
	 * @return true si le dictionnaire a été créé correctement, false sinon
	 * @since 2.0
	 */
	public static boolean fabriqueDicoNat(String dico, String dicoNat,String encodage)
	{
		boolean retour = true;
		if(encodage==null){encodage="UTF-8";}
		BufferedWriter bw = null;
		ArrayList<String> listes = new ArrayList<String>();
		//lecture et stockage des patterns dans une ArrayList
		boolean fini=false;
		try
		{
			BufferedReader br = new BufferedReader(new InputStreamReader(new FileInputStream(dico),encodage));
			//première ligne donnant encodage ?
			String l1 = br.readLine();
			if(!l1.toLowerCase().equals(l1))//c'est pas une règle a priori
			{
				br = new BufferedReader(new InputStreamReader(new FileInputStream(dico),l1));
				br.readLine();//on lit la 1ère ligne de nouveau pour la virer...
			}
			else{br = new BufferedReader(new InputStreamReader(new FileInputStream(dico),encodage));}
			//lecture 
			while(!fini)
			{
				String lu = br.readLine();
				if(lu==null){fini=true;}
				else
				{
					if(lu.startsWith(".")){lu = "^" + lu.substring(1);}
					if(lu.endsWith(".")){lu = lu.substring(0, lu.lastIndexOf("."))+"$";}
					lu = lu.replace("'", "''");
					listes.add(lu);
				}
			}
			br.close();
			//écriture du fichier xsl contenant le paramètre patterns d'après la liste précédemment chargée
			bw=new BufferedWriter(new OutputStreamWriter(new FileOutputStream(dicoNat),encodage));
			//entêtes
			bw.write("<?xml version=\"1.1\" encoding=\"UTF-8\"?>\n" +
					"<!DOCTYPE xsl:stylesheet SYSTEM \"file://"+
					ConfigNat.getCurrentConfig().getDTD()+"\">\n" +
				"<xsl:stylesheet version=\"2.0\"\n"+
				"xmlns:xsl='http://www.w3.org/1999/XSL/Transform'\n"+ 
				"xmlns:xs='http://www.w3.org/2001/XMLSchema'>\n"+
				"<!-- auto-generated by NAT from "+ dico +"-->\n" +
				"<xsl:include href=\""+ConfigNat.getInstallFolder()+"xsl/hyphenation.xsl\""+"/>\n"+
				"<xsl:param name=\"brailleHyphen\" as=\"xs:boolean\" select=\"true()\"/>\n");
			//variable des entités braille
			bw.write("<xsl:variable name=\"v\" as=\"xs:string\">" +
				"&pt1;&pt12;&pt14;&pt145;&pt15;&pt124;&pt1245;&pt125;&pt24;&pt245;" +
				"&pt13;&pt123;&pt134;&pt1345;&pt135;&pt1234;&pt12345;&pt1235;&pt234;&pt2345;" +
				"&pt136;&pt1236;&pt2456;&pt1346;&pt13456;&pt1356;&pt12356;&pt16;&pt123456;&pt2346;" +
				"&pt126;&pt1246;&pt146;&pt12456;&pt1456;&pt23456;&pt156;&pt1256;&pt12346;&pt345;" +
				"&pt246;&pt3;&pt36;</xsl:variable>\n");
			//les patterns
			bw.write("<xsl:param name=\"patterns\" as=\"xs:string*\" select=\"if ($brailleHyphen) then " +
					"tokenize(translate('");
			String patterns = listes.get(0);
			for (int i=1;i<listes.size();i++)
			{
				patterns = patterns +"," + listes.get(i);				
			}
			bw.write(patterns+"','abcdefghijklmnopqrstuvwxyzàâéèêëîïôùûüçæœ''-',$v),',')\n" +
					"else tokenize('" + patterns + "',',')\"/>");
			//la fin de la feuille
			bw.write("\n</xsl:stylesheet>");
			bw.close();
		}
		catch(FileNotFoundException fnfe){fnfe.printStackTrace();retour=false;}
		catch (IOException ioe){ioe.printStackTrace();retour=false;}
		return retour;
	}
	/**
	 * Renvoie une liste double contenant les règles de coupures de nat regroupées par longueur de règle
	 * <p>La liste renvoyée contient des sous-listes dont les règles sont de même longueur pour chaque sous-liste</p> 
	 * @param dicoNat l'adresse du dictionnaire de coupure
	 * @return la double liste conteannt les règles de coupure
	 * @deprecated a priori ne sert plus car le critère de longueur de règle ne sert pas dans les 
	 * règles de coupures
	 * @since 1.5
	 */
	@Deprecated
	public static ArrayList<ArrayList<String>> getRulesOrdered(String dicoNat)
	{
		ArrayList<ArrayList<String>> retour = new ArrayList<ArrayList<String>>();
		boolean fini=false;
		try
		{
			BufferedReader br = new BufferedReader(new InputStreamReader(new FileInputStream(dicoNat),encodageDefaut));
			while(!fini)
			{
				String lu = br.readLine();
				if(lu==null){fini=true;}
				else
				{
					int place = lu.indexOf('C');
					if(retour.size()<=place)
					{
						for(int i=retour.size(); i< place+1; i++){retour.add(new ArrayList<String>());}
					}
					retour.get(place).add(lu);
				}
			}
			br.close();
		}
		catch(FileNotFoundException fnfe){fnfe.printStackTrace();retour=null;}
		catch (IOException ioe){ioe.printStackTrace();retour=null;}
		return retour;
	}
	/**
	 * Renvoie une liste contenant les règles de coupures de nat
	 * <p>La liste renvoyée est ordonnée telle que dans le fichier de règle</p>
	 * <p>Adapte les patterns pour les expressions régulières en xsl:</p>
	 * <ul>
	 * 	<li>remplacement des . du début par ^</li>
	 * 	<li>remplacement des . de fin par $</li>
	 * 	<li>doublement des apostrophes</li>
	 * </ul>
	 * <p>utilise l'encodage par défaut ({@link #encodageDefaut})SAUF si un encodage est précisé dans le fichier de règles</p>
	 * @param dico l'adresse du dictionnaire utilisé, au format compatible avec
	 * l'algorithme de Liang(Tex)
	 * @return la liste contenant les règles de coupure
	 * @see #getRules(String, String)
	 * @since 2.0
	 */
	public static ArrayList<String> getRules(String dico)
	{
		return getRules(dico,encodageDefaut);
	}
	/**
	 * Renvoie une liste contenant les règles de coupures de nat
	 * <p>La liste renvoyée est ordonnée telle que dans le fichier de règles.</p>
	 * <p>Adapte les patterns pour les expressions régulières en xsl:</p>
	 * <ul>
	 * 	<li>remplacement des . du début par ^</li>
	 * 	<li>remplacement des . de fin par $</li>
	 * 	<li>doublement des apostrophes</li>
	 * </ul>
	 * <p>utilise l'encodage spécifié par <code>encodage</code> 
	 * ou {@link #encodageDefaut} si <code>encodage</code> vaut <code>null</code>
	 * SAUF si un encodage est précisé dans le fichier de règles à la première ligne</p>
	 * @param dico l'adresse du dictionnaire utilisé, au format compatible avec
	 * l'algorithme de Liang(Tex)
	 * @param encodage l'encodage du fichier de règle. Si <code>null</code>,
	 * utilisation de l'encodage par défaut ({@link #encodageDefaut}) 
	 * @return la liste contenant les règles de coupure
	 * @since 2.0
	 */
	public static ArrayList<String> getRules(String dico, String encodage)
	{
		if(encodage==null){encodage = encodageDefaut;}
		ArrayList<String> retour = new ArrayList<String>();
		boolean fini=false;
		try
		{
			BufferedReader br = new BufferedReader(new InputStreamReader(new FileInputStream(dico),encodage));
			//première ligne donnant encodage ?
			String l1 = br.readLine(); 
			if(!l1.toLowerCase().equals(l1))//c'est pas une règle a priori car elle contient des maj
			{
				encodage = l1;//comme il est spécifié, on utilise l'encodage écrit dans le fichier de règles
				br = new BufferedReader(new InputStreamReader(new FileInputStream(dico),l1));
				br.readLine();//on lit la 1ère ligne de nouveau pour la virer...
			}
			else{br = new BufferedReader(new InputStreamReader(new FileInputStream(dico),encodage));}
			//lecture 
			while(!fini)
			{
				String lu = br.readLine();
				if(lu==null){fini=true;}
				/* mise sous forme de regexp compatible xsl */
				else
				{
					if(lu.startsWith(".")){lu = "^" + lu.substring(1);}
					if(lu.endsWith(".")){lu = lu.substring(0, lu.lastIndexOf("."))+"$";}
					lu = lu.replace("'", "''");
					retour.add(lu);
				}
			}
			br.close();
		}
		catch(FileNotFoundException fnfe){fnfe.printStackTrace();retour=null;}
		catch (IOException ioe){ioe.printStackTrace();retour=null;}
		return retour;
	}
	/**
	 * Ecrit les règles contenus dans <code>rules</code> dans le fichier <code>dico</code>
	 * en utilisant l'encodage {@link #encodageDefaut}.
	 * <p>Les règles sont sous la forme de regex compatibles avec xsl. Cette fonction remplace:</p>
	 * <ul>
	 * 	<li>les ^ du début par .</li>
	 * 	<li>les $ de fin par .</li>
	 * 	<li>les apostrophes doublées par une apostrophe.</li>
	 * </ul>
	 * @param rules ArrayList des règles au format xsl-compatible
	 * @param dico L'adresse du fichier dictionnaire
	 * @return <code>true</code> si l'écriture dans <code>dico</code> s'est bien déroulée
	 * @see #writeRules(ArrayList, String, String)
	 * @since 2.0
	 */
	public static boolean writeRules(ArrayList<String> rules,String dico)
	{
		return writeRules(rules,dico,encodageDefaut);
	}
	/**
	 * Ecrit les règles contenus dans <code>rules</code> dans le fichier <code>dico</code>
	 * en utilisant l'encodage <code>encodage</code>.
	 * <p>Les règles sont sous la forme de regex compatibles avec xsl. Cette fonction remplace:</p>
	 * <ul>
	 * 	<li>les ^ du début par .</li>
	 * 	<li>les $ de fin par .</li>
	 * 	<li>les apostrophes doublées par une apostrophe.</li>
	 * </ul>
	 * @param dico ArrayList des règles au format xsl-compatible
	 * @param encodage Encodage du fichier <code>dico</code>
	 * @param rules L'adresse du fichier dictionnaire
	 * @return <code>true</code> si l'écriture dans <code>dico</code> s'est bien déroulée
	 * @since 2.0
	 */
	public static boolean writeRules(ArrayList<String> rules, String dico, String encodage)
	{
		if(encodage==null){encodage = encodageDefaut;}
		boolean retour = true;
		try
		{
			BufferedWriter bw = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(dico),encodage));
			//la première ligne donne l'encodage
			bw.write(encodage.toUpperCase()+"\n");
			
			//écriture 
			for(int i=0;i<rules.size();i++)
			{
				String rule = rules.get(i);
				/* mise sous forme de règle compatible tex/liang */
				if(rule.startsWith("^")){rule = "." + rule.substring(1);}
				if(rule.endsWith("$")){rule = rule.substring(0, rule.lastIndexOf("$"))+".";}
				rule = rule.replace("''", "'");
				bw.write(rule+"\n");
			}
			bw.close();
		}
		catch(FileNotFoundException fnfe){fnfe.printStackTrace();retour=false;}
		catch (IOException ioe){ioe.printStackTrace();retour=false;}
		return retour;
	}
	/**
	 * méthode de test
	 * @param a unused
	 */
	public static void main(String []a)
	{
		HyphenationToolkit.fabriqueDicoNat("xsl/dicts/hyph_fr.dic", "xsl/dicts/testHyphdic2.dic", "utf-8");
	}
}