/*
 * Mimic apstat -n / -no
 */
#include "common.h"

/**
 * struct node_information  -  information to be printed out by apstat -n
 */
struct node_information {
	uint32_t		node_id;

	enum basil_node_arch	arch;		/* XT or X2 */
	bool			is_up;		/* node stat: "DN" or "UP" */
	enum basil_node_role	role;		/* allocation mode: B | I  */
	uint64_t		apid;		/* first APID on node      */
	uint8_t			num_apids;	/* number of APIDs on node */

	/*
	 * Counters for cores (processors)
	 */
	uint16_t	ncores,		/* HW:  cores/CPUs on the node		*/
			cores_rsv,	/* Rv:  cores held in a reservation	*/
			cores_use;	/* Pl:  cores in use by an application	*/
	uint16_t	pes;		/* PEs: number of reserved node PEs     */
	/*
	 * Memory counters
	 */
	uint32_t	page_size;	/* PgSz:   page size (4K or 2M)		*/
	uint32_t	page_count,	/* Avl:    number of pages available	*/
			conf,		/* Conf:   total pages confirmed	*/
			placed;		/* Placed: total pages placed		*/
};

/* Sort in ascending order of node ID (default) */
static int cmp_by_nid(const void *a, const void *b)
{
	return	((struct node_information *)a)->node_id >
		((struct node_information *)b)->node_id;
}

/**
 * pes_from_memory_share  -  compute PE share of memory
 */
static uint32_t pes_from_memory_share(uint32_t avail_mem, uint32_t claimed_mem,
				      uint32_t cores_per_node)
{
	if (avail_mem == 0)
		return 0;
#define DIV_ROUND_UP(n,d) (((n) + (d) - 1) / (d))
	return DIV_ROUND_UP((cores_per_node * claimed_mem), avail_mem);
}

/**
 * apstat_n  -  mimic apstat -n / -no
 * @ai_buf:	start of appinfo buffer
 * @inv:	recent ALPS Inventory
 * @verbose:	0 - just print summary
 * 		1 - full listing, ordered by node ID (-n)
 * 		2 - full listing, ordered by ALPS placement order (-no)
 */
void apstat_n(const uint8_t *ai_buf, struct basil_inventory *inv, unsigned verbose)
{
	struct basil_node *node;
	struct node_information *node_array;
	struct node_placement	*np_head, *np_this;
	uint32_t node_count = basil_node_total_count(inv),
		 nodes_config = 0, nodes_up = 0, nodes_inuse = 0,
		 nodes_held = 0, nodes_avail, nodes_down, i;

	if (!node_count)
		return;
	else if (verbose)
#if CLE > 2
		printf("  NID Arch State HW Rv Pl  PgSz      Avl     Conf   Placed  PEs Apids\n");
#else
		printf("  NID Arch State HW Rv Pl  PgSz     Avl    Conf  Placed  PEs Apids\n");
#endif

	np_head	= appinfo_get_node_placements(ai_buf);

	node_array = calloc(node_count, sizeof(*node_array));
	if (node_array == NULL)
		err(1, "can not allocate node_array[%d]", node_count);

	/*  Inventory nodes are in stack order, hence count backwards for -no */
	for (node = inv->f->node_head, i = node_count - 1; node; node = node->next, i--) {
		uint32_t mem_pes;

		node_array[i].node_id	= node->node_id;
		node_array[i].arch	= node->arch;
		node_array[i].is_up	= node->state == BNS_UP;
		node_array[i].role	= node->role;
		node_array[i].page_size	= basil_node_page_size(node);
		node_array[i].page_count= basil_node_page_count(node);
		node_array[i].conf	= basil_node_pages_conf(node);

		node_array[i].ncores	= basil_node_total_cores(node);
		node_array[i].cores_rsv	= basil_node_reserved_cores(node);

		np_this = np_lookup(np_head, node->node_id);
		if (np_this == NULL) {
			node_array[i].num_apids = 0;	/* nothing placed  */
			node_array[i].placed	= 0;	/* no memory usage */
			node_array[i].cores_use	= 0;	/* no cores usage  */
			node_array[i].pes	= 0;	/* no PEs claimed  */
		} else {
			if (np_this->cmd_idx > -1) {
				node_array[i].num_apids	= 1;
				node_array[i].apid	= np_this->apid;
			}

			/* Conf/Placed are in KB, page_size is in units of KB, memory is in MB */
			node_array[i].placed = np_this->memory * 1024 / node_array[i].page_size;

			if (np_this->res_nppn == 0) {
				if (np_this->nppn > np_this->res_nppn)
					node_array[i].cores_rsv	= np_this->depth * np_this->nppn;
				else if (np_this->res_depth > 1)
					node_array[i].cores_rsv = np_this->res_depth;
				else
					node_array[i].cores_rsv = node_array[i].ncores;
			} else if (np_this->res_depth > 1) {
#if 0&& CLE <= 2
				node_array[i].cores_rsv = np_this->res_depth;
#else
				node_array[i].cores_rsv = np_this->res_depth * np_this->res_nppn;
#endif
				node_array[i].pes	= node_array[i].ncores / np_this->res_nppn;
			} else {
				node_array[i].cores_rsv = np_this->res_depth * np_this->res_nppn;
			}

			mem_pes = pes_from_memory_share(node_array[i].page_count,
							node_array[i].conf,
							node_array[i].ncores);

			if (np_this->res_depth == 1 && mem_pes < node_array[i].cores_rsv)
				node_array[i].cores_rsv = mem_pes;

			if (np_this->nppn == 0) {
				if (np_this->depth > 1)
					node_array[i].cores_use = np_this->depth;
				else
					node_array[i].cores_use = node_array[i].ncores;
				node_array[i].pes = node_array[i].cores_use;
			} else {
				node_array[i].cores_use	= np_this->depth * np_this->nppn;
				node_array[i].pes	= np_this->nppn;
			}

			mem_pes = pes_from_memory_share(node_array[i].conf,
							node_array[i].placed,
							node_array[i].cores_rsv);


			/* XXX for debugging XXX */
			if (node->node_id == 1344 && 0)
				printf("Rv=%d Pl=%d mem=%d, Conf=%d, Placed=%d PEs=%d/%d nppn=%d, depth=%d, cores=%d"
					", RES: nppn=%d, depth=%d, mem=%d\n",
					node_array[i].cores_rsv,node_array[i].cores_use, np_this->memory,
					node_array[i].conf, node_array[i].placed,
					node_array[i].pes ,mem_pes,np_this->nppn, np_this->depth, np_this->cores,
					np_this->res_nppn, np_this->res_depth, np_this->res_memory
				),abort();

			if (np_this->memory < np_this->res_memory) {
				node_array[i].pes	= np_this->cores;
				node_array[i].cores_use	= np_this->cores * np_this->depth;
				node_array[i].placed	= np_this->memory * 1024;	/* MB -> KB */
				if (np_this->cores < node_array[i].cores_rsv ||
				    np_this->cores > np_this->res_nppn)
					node_array[i].placed = (node_array[i].placed * np_this->cores)/node_array[i].cores_rsv;
			} else if (mem_pes == 1) {
				if (np_this->depth == 1) {
					if (node_array[i].cores_rsv > np_this->cores) {
						node_array[i].cores_use	= np_this->cores;
						node_array[i].pes	= np_this->cores;
						node_array[i].placed	= DIV_ROUND_UP(node_array[i].conf * node_array[i].pes, node_array[i].cores_rsv);
					} else {
						node_array[i].cores_use	= node_array[i].cores_rsv;
						node_array[i].pes	= node_array[i].cores_rsv;
					}
				} else if (node_array[i].pes > np_this->cores) {
					node_array[i].cores_use	= np_this->depth;
					node_array[i].pes	= np_this->cores;
				}
			} else if (node_array[i].cores_rsv > np_this->cores) {
#if CLE <= 2
				node_array[i].cores_use	= np_this->depth;
#else
				if (np_this->width * np_this->depth == np_this->res_width * np_this->res_depth)
					node_array[i].cores_use	= node_array[i].cores_rsv;
				else
					node_array[i].cores_use	= np_this->cores;
#endif
				node_array[i].pes = np_this->cores;
			}

			if (node_array[i].placed == 0)		/* no memory allocated -> no cores in use */
				node_array[i].pes = node_array[i].cores_use = 0;

			if (node_array[i].cores_use == node_array[i].cores_rsv) {
				if (node_array[i].pes == 1)
					node_array[i].placed = node_array[i].conf;
				else if (np_this->memory == np_this->res_memory)
					node_array[i].placed = DIV_ROUND_UP(node_array[i].conf * node_array[i].pes, node_array[i].cores_rsv);
			}
		}

		//if (node->state != BNS_DOWN)
			nodes_config++;

		if (node_array[i].cores_rsv) {
			if (!node_array[i].cores_use)
				nodes_held++;
			else
				nodes_inuse++;
		}
		nodes_up += node_array[i].is_up;
	}
	assert(i == -1);

	if (verbose == 1)	/* -n instead of -no */
		qsort(node_array, node_count, sizeof(*node_array), cmp_by_nid);

	for (i = verbose ? 0 : node_count; i < node_count; i++) {
		const char *nam_node_state_brief[] = { "DN", "UP" };
		char apids[BASIL_STRING_LONG] = "";

		if (!node_array[i].is_up)	/* DN nodes are not shown */
			continue;

		if (node_array[i].num_apids)
			snprintf(apids, sizeof(apids), "%llu",
				 (unsigned long long)node_array[i].apid);
#if CLE > 2
		printf("%5u %4s %2s  %1.1s %2u %2u %2u %3uK  %8u %8u %8u %4u %s\n",
#else
		printf("%5u %4.4s %2s  %1.1s %2u %2u %2u %4uK %7u %7u %7u %4u %s\n",
#endif
			node_array[i].node_id,
			nam_arch[node_array[i].arch],
			nam_node_state_brief[node_array[i].is_up],
			nam_noderole[node_array[i].role],
			node_array[i].ncores,
			node_array[i].cores_rsv,
			node_array[i].cores_use,
			node_array[i].page_size,
			node_array[i].page_count,
			node_array[i].conf,
			node_array[i].placed,
			node_array[i].pes,
			apids);
	}
	nodes_down  = nodes_config - nodes_up;
	nodes_avail = nodes_config - nodes_down - nodes_inuse - nodes_held;

	printf("Compute node summary\n"
#if CLE > 2
	       "%8s%7s%7s%7s%7s%7s%7s\n"
	       "%8s%7d%7d%7d%7d%7d%7d\n",
#else
	       "%7s%7s%7s%7s%7s%7s%7s\n"
	       "%7s%7d%7d%7d%7d%7d%7d\n",
#endif
	       "arch", "config", "up", "use", "held", "avail", "down",
		nam_arch[node_array[0].arch], nodes_config, nodes_up,
		nodes_inuse, nodes_held, nodes_avail, nodes_down);

	np_free(np_head);
	free(node_array);
}


