# $Id: pep.tcl 1331 2007-12-23 15:21:57Z sergei $
# Personal eventing via pubsub XEP-0163

namespace eval pep {

    custom::defgroup Plugins \
	[::msgcat::mc "Plugins options."] \
	-group Tkabber

    custom::defgroup PEP \
	[::msgcat::mc "Personal eventing via pubsub plugins options."] \
	-group Plugins
}

##########################################################################
#
# PEP Creating a node (5)
# -connection is mandatory
# -access_model (open, presence (default), roster, whitelist)
# -roster_groups_allowed (roster group list if access is roster)
#

proc pep::create_node {node args} {
    variable ns

    debugmsg pep [info level 0]

    set command ""
    set access "presence"
    set groups {}
    foreach {key val} $args {
	switch -- $key {
	    -connection { set connid $val}
	}
    }

    if {![info exists connid]} {
	return -code error "pep::create_node: -connection is mandatory"
    }

    if {$node == ""} {
	return -code error "pep::create_node: node must not be empty"
    }

    set service [jlib::connection_bare_jid $connid]

    eval [list pubsub::create_node $service $node] $args
}

##########################################################################
#
# Publish item to PEP node "node" (8)
# payload is a list of xml tags
# node must not be empty
# itemid may be empty
# -connection is mandatory
#

proc pep::publish_item {node itemid args} {

    debugmsg pep [info level 0]

    set command ""
    foreach {key val} $args {
	switch -- $key {
	    -connection { set connid $val}
	}
    }

    if {![info exists connid]} {
	return -code error "pep::publish_item: -connection is mandatory"
    }

    if {$node == ""} {
	return -code error "pep::publish_item: node must not be empty"
    }

    set service [jlib::connection_bare_jid $connid]

    eval [list pubsub::publish_item $service $node $itemid] $args
}

##########################################################################
#
# Delete item from PEP node "node"
# node must not be empty
# itemid must not be empty
# -connection is mandatory
#

proc pep::delete_item {node itemid args} {

    debugmsg pep [info level 0]

    set command ""
    foreach {key val} $args {
	switch -- $key {
	    -connection { set connid $val}
	}
    }

    if {![info exists connid]} {
	return -code error "pep::delete_item: -connection is mandatory"
    }

    if {$node == ""} {
	return -code error "pep::delete_item: node must not be empty"
    }

    if {$itemid == ""} {
	return -code error "pep::delete_item: Item ID must not be empty"
    }

    set service [jlib::connection_bare_jid $connid]

    eval [list pubsub::delete_item $service $node $itemid] $args
}

##########################################################################
#
# Subscribe to PEP node "node" at bare JID "to" (5.2)
# node must not be empty
#
# -jid "jid" is optional (when it's present it's included to sub request)
#
# -resource "res" is optional (when it's present bare_jid/res is included
# to sub request
#
# if both options are absent then user's bare JID is included to sub
# request
#

proc pep::subscribe {to node args} {

    debugmsg pep [info level 0]

    foreach {key val} $args {
	switch -- $key {
	    -connection { set connid $val}
	}
    }

    if {![info exists connid]} {
	return -code error "pep::subscribe error: -connection is mandatory"
    }

    if {$node == ""} {
	return -code error "pep::subscribe error: node must not be empty"
    }

    eval [list pubsub::subscribe $to $node] $args
}

##########################################################################
#
# Unsubscribe from PEP node "node" at bare JID "to" (undocumented?!)
# node must not be empty
#
# -jid "jid" is optional (when it's present it's included to sub request)
#
# -resource "res" is optional (when it's present bare_jid/res is included
# to sub request
#
# if both options are absent then user's bare JID is included to sub
# request
#

proc pep::unsubscribe {to node args} {

    debugmsg pep [info level 0]

    set command ""
    foreach {key val} $args {
	switch -- $key {
	    -connection { set connid $val}
	}
    }

    if {![info exists connid]} {
	return -code error "pep::unsubscribe error: -connection is mandatory"
    }

    if {$node == ""} {
	return -code error "pep::unsubscribe error: node must not be empty"
    }

    eval [list pubsub::unsubscribe $to $node] $args
}

##########################################################################
# Returns a name of a submenu (of menu $m) for PEP commands to perform on
# the roster item for a user with JID $jid.
# This command automatically creates this submenu if needed.

proc pep::get_roster_menu_pep_submenu {m connid jid} {
    set pm $m.pep

    if {![winfo exists $pm]} {
	menu $pm -tearoff no
	$m add cascade -menu $pm \
		-label [::msgcat::mc "Personal eventing"]
    }

    return $pm
}

##########################################################################
# Returns pathname of a frame comprising a page for PEP info in
# the userinfo (vCard) dialog which notebook widget is $notebook.
# If that page is not yet exist, it's created.

proc pep::get_userinfo_dialog_pep_frame {notebook} {
    if {[$notebook index PEP] < 0} {
	return [$notebook insert end PEP \
			-text [::msgcat::mc "Personal eventing"]]
    } else {
	return [$notebook getframe PEP]
    }
}

proc pep::get_main_menu_pep_submenu {} {
    return [.mainframe getmenu services].pep
}

proc pep::on_init {} {
    set m [.mainframe getmenu services]
    set idx [$m index [::msgcat::mc "Service Discovery"]]
    set pm [menu $m.pep -tearoff $::ifacetk::options(show_tearoffs)]
    $m insert [expr {$idx + 2}] cascade -menu $pm \
	    -label [::msgcat::mc "Personal eventing"]
}

hook::add finload_hook [namespace current]::pep::on_init

# vim:ts=8:sw=4:sts=4:noet
